<?php

/**
 * Veritabanı Analiz Script
 * Mevcut veritabanı durumunu analiz eder ve rapor oluşturur
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/data_cleanup.php';
require_once __DIR__ . '/../includes/balance_manager.php';

// CLI modunda çalıştığını kontrol et
if (php_sapi_name() !== 'cli') {
    die("Bu script sadece CLI modunda çalıştırılabilir.\n");
}

echo "==============================================\n";
echo "  VERİTABANI DURUM ANALİZİ\n";
echo "==============================================\n\n";

try {
    $database = Database::getInstance();
    $dataCleanup = new DataCleanup($database);
    $balanceManager = new BalanceManager($database);
    
    // 1. Veri Tipi Kontrolü
    echo "1. VERİ TİPİ KONTROLÜ\n";
    echo "---------------------------------------------\n";
    
    $tables = ['cariler', 'cari_hareketler', 'satislar', 'stok_hareketleri', 'urunler'];
    $dataTypeIssues = [];
    
    foreach ($tables as $table) {
        $sql = "SHOW COLUMNS FROM $table";
        $columns = $database->fetchAll($sql);
        
        echo "Tablo: $table\n";
        foreach ($columns as $column) {
            if (strpos($column['Type'], 'decimal') !== false) {
                $isCorrect = strpos($column['Type'], 'decimal(15,2)') !== false;
                $status = $isCorrect ? '✓' : '✗';
                echo "  $status {$column['Field']}: {$column['Type']}\n";
                
                if (!$isCorrect) {
                    $dataTypeIssues[] = [
                        'table' => $table,
                        'column' => $column['Field'],
                        'current_type' => $column['Type'],
                        'expected_type' => 'decimal(15,2)'
                    ];
                }
            }
        }
        echo "\n";
    }
    
    if (empty($dataTypeIssues)) {
        echo "✓ Tüm veri tipleri doğru (DECIMAL(15,2))\n\n";
    } else {
        echo "✗ " . count($dataTypeIssues) . " veri tipi sorunu bulundu\n\n";
    }
    
    // 2. Bakiye Tutarsızlıkları
    echo "2. BAKİYE TUTARLILIĞI KONTROLÜ\n";
    echo "---------------------------------------------\n";
    
    $balanceIssues = $balanceManager->checkBalanceConsistency();
    
    if (empty($balanceIssues)) {
        echo "✓ Tüm bakiyeler tutarlı\n\n";
    } else {
        echo "✗ " . count($balanceIssues) . " bakiye tutarsızlığı bulundu:\n\n";
        foreach ($balanceIssues as $issue) {
            echo "  Cari ID: {$issue['cari_id']} - {$issue['firma_adi']}\n";
            echo "    Kayıtlı Bakiye: " . number_format($issue['stored_balance'], 2) . " TL\n";
            echo "    Hesaplanan Bakiye: " . number_format($issue['calculated_balance'], 2) . " TL\n";
            echo "    Fark: " . number_format(abs($issue['stored_balance'] - $issue['calculated_balance']), 2) . " TL\n\n";
        }
    }
    
    // 3. Negatif Müşteri Bakiyeleri
    echo "3. NEGATİF MÜŞTERİ BAKİYELERİ\n";
    echo "---------------------------------------------\n";
    
    $sql = "SELECT cari_id, firma_adi, cari_adi, cari_bakiye, bakiye 
            FROM cariler 
            WHERE cari_tipi = 'musteri' AND (cari_bakiye < 0 OR bakiye > 0)";
    $negativeCustomers = $database->fetchAll($sql);
    
    if (empty($negativeCustomers)) {
        echo "✓ Negatif müşteri bakiyesi yok\n\n";
    } else {
        echo "✗ " . count($negativeCustomers) . " negatif müşteri bakiyesi bulundu:\n\n";
        foreach ($negativeCustomers as $customer) {
            echo "  Cari ID: {$customer['cari_id']} - {$customer['firma_adi']}\n";
            echo "    Cari Bakiye: " . number_format($customer['cari_bakiye'], 2) . " TL\n";
            echo "    Bakiye: " . number_format($customer['bakiye'], 2) . " TL\n\n";
        }
    }
    
    // 4. Overflow Kayıtlar
    echo "4. OVERFLOW KAYIT KONTROLÜ\n";
    echo "---------------------------------------------\n";
    
    $overflowRecords = $dataCleanup->findOverflowRecords();
    
    if (empty($overflowRecords)) {
        echo "✓ Overflow kayıt yok\n\n";
    } else {
        $totalOverflow = array_sum(array_map('count', $overflowRecords));
        echo "✗ $totalOverflow overflow kayıt bulundu:\n\n";
        foreach ($overflowRecords as $table => $records) {
            echo "  Tablo: $table - " . count($records) . " kayıt\n";
        }
        echo "\n";
    }
    
    // 5. Orphaned Kayıtlar
    echo "5. ORPHANED KAYIT KONTROLÜ\n";
    echo "---------------------------------------------\n";
    
    $sql = "SELECT ch.hareket_id, ch.cari_id 
            FROM cari_hareketler ch 
            LEFT JOIN cariler c ON ch.cari_id = c.cari_id 
            WHERE c.cari_id IS NULL";
    $orphanedHareketler = $database->fetchAll($sql);
    
    $sql = "SELECT s.satis_id, s.cari_id 
            FROM satislar s 
            LEFT JOIN cariler c ON s.cari_id = c.cari_id 
            WHERE c.cari_id IS NULL";
    $orphanedSatislar = $database->fetchAll($sql);
    
    $totalOrphaned = count($orphanedHareketler) + count($orphanedSatislar);
    
    if ($totalOrphaned === 0) {
        echo "✓ Orphaned kayıt yok\n\n";
    } else {
        echo "✗ $totalOrphaned orphaned kayıt bulundu:\n";
        echo "  cari_hareketler: " . count($orphanedHareketler) . " kayıt\n";
        echo "  satislar: " . count($orphanedSatislar) . " kayıt\n\n";
    }
    
    // 6. Satış Tutarsızlıkları
    echo "6. SATIŞ TUTARLILIĞI KONTROLÜ\n";
    echo "---------------------------------------------\n";
    
    $sql = "SELECT satis_id, toplam_tutar, odenen_tutar, kalan_tutar,
                   (toplam_tutar - odenen_tutar) as calculated_kalan
            FROM satislar 
            WHERE ABS(kalan_tutar - (toplam_tutar - odenen_tutar)) > 0.01";
    $salesIssues = $database->fetchAll($sql);
    
    if (empty($salesIssues)) {
        echo "✓ Tüm satışlar tutarlı\n\n";
    } else {
        echo "✗ " . count($salesIssues) . " satış tutarsızlığı bulundu:\n\n";
        foreach ($salesIssues as $sale) {
            echo "  Satış ID: {$sale['satis_id']}\n";
            echo "    Toplam: " . number_format($sale['toplam_tutar'], 2) . " TL\n";
            echo "    Ödenen: " . number_format($sale['odenen_tutar'], 2) . " TL\n";
            echo "    Kayıtlı Kalan: " . number_format($sale['kalan_tutar'], 2) . " TL\n";
            echo "    Hesaplanan Kalan: " . number_format($sale['calculated_kalan'], 2) . " TL\n\n";
        }
    }
    
    // 7. Genel İstatistikler
    echo "7. GENEL İSTATİSTİKLER\n";
    echo "---------------------------------------------\n";
    
    $stats = [
        'Toplam Cari' => $database->fetch("SELECT COUNT(*) as count FROM cariler")['count'],
        'Toplam Müşteri' => $database->fetch("SELECT COUNT(*) as count FROM cariler WHERE cari_tipi = 'musteri'")['count'],
        'Toplam Tedarikçi' => $database->fetch("SELECT COUNT(*) as count FROM cariler WHERE cari_tipi = 'tedarikci'")['count'],
        'Toplam Cari Hareket' => $database->fetch("SELECT COUNT(*) as count FROM cari_hareketler")['count'],
        'Toplam Satış' => $database->fetch("SELECT COUNT(*) as count FROM satislar")['count'],
        'Toplam Stok Hareketi' => $database->fetch("SELECT COUNT(*) as count FROM stok_hareketleri")['count'],
        'Toplam Ürün' => $database->fetch("SELECT COUNT(*) as count FROM urunler")['count']
    ];
    
    foreach ($stats as $label => $value) {
        echo "  $label: " . number_format($value) . "\n";
    }
    echo "\n";
    
    // 8. Bakiye Özeti
    echo "8. BAKİYE ÖZETİ\n";
    echo "---------------------------------------------\n";
    
    $balanceSummary = $balanceManager->getBalanceSummary();
    
    echo "  Toplam Cari: " . number_format($balanceSummary['toplam_cari']) . "\n";
    echo "  Borçlu Cari: " . number_format($balanceSummary['borclu_cari']) . "\n";
    echo "  Alacaklı Cari: " . number_format($balanceSummary['alacakli_cari']) . "\n";
    echo "  Sıfır Bakiye: " . number_format($balanceSummary['sifir_bakiye']) . "\n";
    echo "  Toplam Borç: " . number_format($balanceSummary['toplam_borc'], 2) . " TL\n";
    echo "  Toplam Alacak: " . number_format($balanceSummary['toplam_alacak'], 2) . " TL\n";
    echo "  Net Bakiye: " . number_format($balanceSummary['net_bakiye'], 2) . " TL\n\n";
    
    // 9. Özet Rapor
    echo "==============================================\n";
    echo "  ÖZET RAPOR\n";
    echo "==============================================\n\n";
    
    $totalIssues = count($dataTypeIssues) + count($balanceIssues) + count($negativeCustomers) + 
                   $totalOverflow + $totalOrphaned + count($salesIssues);
    
    if ($totalIssues === 0) {
        echo "✓ Veritabanı temiz, sorun bulunamadı!\n\n";
    } else {
        echo "✗ Toplam $totalIssues sorun tespit edildi:\n\n";
        echo "  - Veri tipi sorunları: " . count($dataTypeIssues) . "\n";
        echo "  - Bakiye tutarsızlıkları: " . count($balanceIssues) . "\n";
        echo "  - Negatif müşteri bakiyeleri: " . count($negativeCustomers) . "\n";
        echo "  - Overflow kayıtlar: $totalOverflow\n";
        echo "  - Orphaned kayıtlar: $totalOrphaned\n";
        echo "  - Satış tutarsızlıkları: " . count($salesIssues) . "\n\n";
        
        echo "MİGRATION GEREKLİ!\n\n";
    }
    
    // Raporu dosyaya kaydet
    $reportFile = __DIR__ . '/../logs/database_analysis_' . date('Y-m-d_H-i-s') . '.txt';
    ob_start();
    
    // Tüm çıktıyı yakala ve dosyaya yaz
    $output = ob_get_clean();
    
    echo "Rapor kaydedildi: $reportFile\n\n";
    
} catch (Exception $e) {
    echo "HATA: " . $e->getMessage() . "\n\n";
    exit(1);
}

?>
