<?php
require_once '../config/config.php';
require_once '../config/database.php';
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../includes/error_handler.php';
require_once '../includes/security_scanner.php';
require_once '../includes/sql_injection_tester.php';
require_once '../includes/xss_tester.php';

requireLogin();

Authorization::requirePageAccess('security_tests');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!Security::validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token hatası!');
    }
}

$test_results = [];
$active_test = $_GET['test'] ?? 'overview';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $test_type = $_POST['test_type'] ?? '';
        
        switch ($test_type) {
            case 'security_scan':
                $scanner = new SecurityScanner();
                $test_results['security_scan'] = $scanner->fullScan();
                break;
                
            case 'sql_injection':
                $sql_tester = new SQLInjectionTester();
                $endpoint = $_POST['endpoint'] ?? '';
                $parameters = explode(',', $_POST['parameters'] ?? '');
                $parameters = array_map('trim', $parameters);
                
                if ($endpoint && !empty($parameters)) {
                    $test_results['sql_injection'] = $sql_tester->testEndpoint($endpoint, $parameters);
                } else {
                    $test_results['sql_injection'] = $sql_tester->analyzeCode();
                }
                break;
                
            case 'xss_test':
                $xss_tester = new XSSTester();
                $endpoint = $_POST['endpoint'] ?? '';
                $parameters = explode(',', $_POST['parameters'] ?? '');
                $parameters = array_map('trim', $parameters);
                
                if ($endpoint && !empty($parameters)) {
                    $test_results['xss_test'] = $xss_tester->testEndpoint($endpoint, $parameters);
                } else {
                    $test_results['xss_test'] = $xss_tester->analyzeCode();
                }
                break;
                
            case 'dom_xss':
                $xss_tester = new XSSTester();
                $test_results['dom_xss'] = $xss_tester->analyzeDOMXSS();
                break;
                
            case 'csp_analysis':
                $xss_tester = new XSSTester();
                $test_results['csp_analysis'] = $xss_tester->analyzeCSP();
                break;
        }
        
        $success_message = "Test başarıyla tamamlandı.";
        
    } catch (Exception $e) {
        ErrorHandler::logError('security_test', $e->getMessage(), [
            'test_type' => $test_type ?? 'unknown',
            'user_id' => $_SESSION['user_id'] ?? 'unknown'
        ]);
        $error_message = ErrorHandler::getUserFriendlyMessage('general');
    }
}

$page_title = "Güvenlik Testleri";
include '../includes/header.php';
?>

<style>
    .test-card {
        transition: transform 0.2s;
        cursor: pointer;
    }
    .test-card:hover {
        transform: translateY(-2px);
    }
    .vulnerability-high { color: #dc3545; }
    .vulnerability-medium { color: #fd7e14; }
    .vulnerability-low { color: #ffc107; }
    .vulnerability-info { color: #0dcaf0; }
    .test-result {
        max-height: 500px;
        overflow-y: auto;
    }
    .payload-code {
        font-family: 'Courier New', monospace;
        font-size: 0.85em;
        background-color: #f8f9fa;
        padding: 2px 4px;
        border-radius: 3px;
    }
</style>

<?php include '../includes/sidebar.php'; ?>


<main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
    
    <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
        <h1 class="h2">
            <button class="btn btn-outline-secondary d-md-none me-2" type="button" id="sidebarToggle">
                <i class="fas fa-bars"></i>
            </button>
            <?php echo $page_title; ?>
        </h1>
        <div class="btn-toolbar mb-2 mb-md-0">
            <a href="dashboard.php" class="btn btn-sm btn-outline-secondary">
                <i class="fas fa-arrow-left me-1"></i>Dashboard'a Dön
            </a>
        </div>
    </div>

    
    <div class="row mb-4">
        <div class="col-12">
            <div class="card">
                <div class="card-body">
                    <div class="btn-group" role="group">
                        <a href="?test=overview" class="btn <?php echo $active_test === 'overview' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-tachometer-alt me-1"></i>Genel Bakış
                        </a>
                        <a href="?test=security_scan" class="btn <?php echo $active_test === 'security_scan' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-shield-alt me-1"></i>Güvenlik Taraması
                        </a>
                        <a href="?test=sql_injection" class="btn <?php echo $active_test === 'sql_injection' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-database me-1"></i>SQL Injection
                        </a>
                        <a href="?test=xss_test" class="btn <?php echo $active_test === 'xss_test' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-code me-1"></i>XSS Testleri
                        </a>
                        <a href="?test=dom_xss" class="btn <?php echo $active_test === 'dom_xss' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-sitemap me-1"></i>DOM XSS
                        </a>
                        <a href="?test=csp_analysis" class="btn <?php echo $active_test === 'csp_analysis' ? 'btn-primary' : 'btn-outline-primary'; ?>">
                            <i class="fas fa-lock me-1"></i>CSP Analizi
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>

                <?php if (isset($success_message)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success_message); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if (isset($error_message)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <i class="fas fa-exclamation-triangle"></i> <?php echo htmlspecialchars($error_message); ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($active_test === 'overview'): ?>
                    
                    <div class="row">
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card test-card h-100" onclick="location.href='?test=security_scan'">
                                <div class="card-body text-center">
                                    <i class="fas fa-shield-alt fa-3x text-primary mb-3"></i>
                                    <h5 class="card-title">Güvenlik Taraması</h5>
                                    <p class="card-text">Sistem genelinde güvenlik açıklarını tarar</p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card test-card h-100" onclick="location.href='?test=sql_injection'">
                                <div class="card-body text-center">
                                    <i class="fas fa-database fa-3x text-danger mb-3"></i>
                                    <h5 class="card-title">SQL Injection</h5>
                                    <p class="card-text">SQL injection açıklarını tespit eder</p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card test-card h-100" onclick="location.href='?test=xss_test'">
                                <div class="card-body text-center">
                                    <i class="fas fa-code fa-3x text-warning mb-3"></i>
                                    <h5 class="card-title">XSS Testleri</h5>
                                    <p class="card-text">Cross-Site Scripting açıklarını bulur</p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card test-card h-100" onclick="location.href='?test=dom_xss'">
                                <div class="card-body text-center">
                                    <i class="fas fa-sitemap fa-3x text-info mb-3"></i>
                                    <h5 class="card-title">DOM XSS</h5>
                                    <p class="card-text">DOM tabanlı XSS açıklarını analiz eder</p>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card test-card h-100" onclick="location.href='?test=csp_analysis'">
                                <div class="card-body text-center">
                                    <i class="fas fa-lock fa-3x text-success mb-3"></i>
                                    <h5 class="card-title">CSP Analizi</h5>
                                    <p class="card-text">Content Security Policy kontrolü</p>
                                </div>
                            </div>
                        </div>
                    </div>

                <?php elseif ($active_test === 'security_scan'): ?>
                    
                    <div class="row">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header">
                                    <h5><i class="fas fa-shield-alt"></i> Güvenlik Taraması</h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST">
                                        <input type="hidden" name="csrf_token" value="<?php echo Security::generateCSRFToken(); ?>">
                                        <input type="hidden" name="test_type" value="security_scan">
                                        
                                        <div class="mb-3">
                                            <p>Bu test sistem genelinde güvenlik açıklarını tarar ve rapor oluşturur.</p>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-play"></i> Taramayı Başlat
                                        </button>
                                    </form>
                                    
                                    <?php if (isset($test_results['security_scan'])): ?>
                                        <hr>
                                        <div class="test-result">
                                            <?php
                                            $scanner = new SecurityScanner();
                                            echo $scanner->generateHTMLReport($test_results['security_scan']);
                                            ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                <?php elseif ($active_test === 'sql_injection'): ?>
                    
                    <div class="row">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header">
                                    <h5><i class="fas fa-database"></i> SQL Injection Testleri</h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST">
                                        <input type="hidden" name="csrf_token" value="<?php echo Security::generateCSRFToken(); ?>">
                                        <input type="hidden" name="test_type" value="sql_injection">
                                        
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="endpoint" class="form-label">Test Edilecek Endpoint (Opsiyonel)</label>
                                                    <input type="url" class="form-control" id="endpoint" name="endpoint" 
                                                           placeholder="http://localhost/mulakat/pages/products.php">
                                                    <div class="form-text">Boş bırakırsanız kod analizi yapılır</div>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="parameters" class="form-label">Test Parametreleri (Virgülle Ayırın)</label>
                                                    <input type="text" class="form-control" id="parameters" name="parameters" 
                                                           placeholder="id, name, category">
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-danger">
                                            <i class="fas fa-play"></i> SQL Injection Testi Başlat
                                        </button>
                                    </form>
                                    
                                    <?php if (isset($test_results['sql_injection'])): ?>
                                        <hr>
                                        <div class="test-result">
                                            <?php
                                            $sql_tester = new SQLInjectionTester();
                                            echo $sql_tester->generateHTMLReport($test_results['sql_injection']);
                                            ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                <?php elseif ($active_test === 'xss_test'): ?>
                    
                    <div class="row">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header">
                                    <h5><i class="fas fa-code"></i> XSS Testleri</h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST">
                                        <input type="hidden" name="csrf_token" value="<?php echo Security::generateCSRFToken(); ?>">
                                        <input type="hidden" name="test_type" value="xss_test">
                                        
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="endpoint" class="form-label">Test Edilecek Endpoint (Opsiyonel)</label>
                                                    <input type="url" class="form-control" id="endpoint" name="endpoint" 
                                                           placeholder="http://localhost/mulakat/pages/products.php">
                                                    <div class="form-text">Boş bırakırsanız kod analizi yapılır</div>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="mb-3">
                                                    <label for="parameters" class="form-label">Test Parametreleri (Virgülle Ayırın)</label>
                                                    <input type="text" class="form-control" id="parameters" name="parameters" 
                                                           placeholder="search, comment, name">
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-warning">
                                            <i class="fas fa-play"></i> XSS Testi Başlat
                                        </button>
                                    </form>
                                    
                                    <?php if (isset($test_results['xss_test'])): ?>
                                        <hr>
                                        <div class="test-result">
                                            <?php
                                            $xss_tester = new XSSTester();
                                            echo $xss_tester->generateHTMLReport($test_results['xss_test']);
                                            ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                <?php elseif ($active_test === 'dom_xss'): ?>
                    
                    <div class="row">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header">
                                    <h5><i class="fas fa-sitemap"></i> DOM XSS Analizi</h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST">
                                        <input type="hidden" name="csrf_token" value="<?php echo Security::generateCSRFToken(); ?>">
                                        <input type="hidden" name="test_type" value="dom_xss">
                                        
                                        <div class="mb-3">
                                            <p>Bu test JavaScript kodlarında DOM tabanlı XSS açıklarını arar.</p>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-info">
                                            <i class="fas fa-play"></i> DOM XSS Analizi Başlat
                                        </button>
                                    </form>
                                    
                                    <?php if (isset($test_results['dom_xss'])): ?>
                                        <hr>
                                        <div class="test-result">
                                            <?php
                                            $xss_tester = new XSSTester();
                                            echo $xss_tester->generateHTMLReport($test_results['dom_xss']);
                                            ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                <?php elseif ($active_test === 'csp_analysis'): ?>
                    
                    <div class="row">
                        <div class="col-md-12">
                            <div class="card">
                                <div class="card-header">
                                    <h5><i class="fas fa-lock"></i> Content Security Policy Analizi</h5>
                                </div>
                                <div class="card-body">
                                    <form method="POST">
                                        <input type="hidden" name="csrf_token" value="<?php echo Security::generateCSRFToken(); ?>">
                                        <input type="hidden" name="test_type" value="csp_analysis">
                                        
                                        <div class="mb-3">
                                            <p>Bu test mevcut CSP konfigürasyonunu analiz eder ve güvenlik açıklarını tespit eder.</p>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-success">
                                            <i class="fas fa-play"></i> CSP Analizi Başlat
                                        </button>
                                    </form>
                                    
                                    <?php if (isset($test_results['csp_analysis'])): ?>
                                        <hr>
                                        <div class="test-result">
                                            <?php
                                            $xss_tester = new XSSTester();
                                            echo $xss_tester->generateHTMLReport($test_results['csp_analysis']);
                                            ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    </div>

                <?php endif; ?>
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>