<?php
require_once '../config/config.php';
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../config/database.php';

requireLogin();

Authorization::requirePageAccess('products', 'okuma');

$error_message = '';
$success_message = '';
$action = $_GET['action'] ?? 'list';
$product_id = $_GET['id'] ?? null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        validateCSRF();
        
        Authorization::requirePageAccess('products', 'yazma');
        
        $urun_adi = Security::sanitizeInput($_POST['urun_adi'] ?? '');
        $barkod = Security::sanitizeInput($_POST['barkod'] ?? '');
        $stok_miktari = (int)($_POST['stok_miktari'] ?? 0);
        $alis_fiyati = (float)($_POST['alis_fiyati'] ?? 0);
        $satis_fiyati = (float)($_POST['satis_fiyati'] ?? 0);
        
        if (empty($urun_adi)) {
            throw new Exception('Ürün adı gereklidir.');
        }
        if ($alis_fiyati < 0 || $satis_fiyati < 0) {
            throw new Exception('Fiyatlar negatif olamaz.');
        }
        if ($stok_miktari < 0) {
            throw new Exception('Stok miktarı negatif olamaz.');
        }
        
        if ($action === 'add') {
            if (!empty($barkod)) {
                $sql = "SELECT COUNT(*) as count FROM urunler WHERE barkod = ?";
                $result = $database->fetch($sql, [$barkod]);
                if ($result['count'] > 0) {
                    throw new Exception('Bu barkod zaten kullanılıyor.');
                }
            }
            
            $sql = "INSERT INTO urunler (urun_adi, barkod, stok_miktari, alis_fiyati, satis_fiyati) VALUES (?, ?, ?, ?, ?)";
            $database->execute($sql, [$urun_adi, $barkod, $stok_miktari, $alis_fiyati, $satis_fiyati]);
            $success_message = 'Ürün başarıyla eklendi.';
            
        } elseif ($action === 'edit' && $product_id) {
            if (!empty($barkod)) {
                $sql = "SELECT COUNT(*) as count FROM urunler WHERE barkod = ? AND urun_id != ?";
                $result = $database->fetch($sql, [$barkod, $product_id]);
                if ($result['count'] > 0) {
                    throw new Exception('Bu barkod zaten kullanılıyor.');
                }
            }
            
            $sql = "UPDATE urunler SET urun_adi = ?, barkod = ?, stok_miktari = ?, alis_fiyati = ?, satis_fiyati = ? WHERE urun_id = ?";
            $database->execute($sql, [$urun_adi, $barkod, $stok_miktari, $alis_fiyati, $satis_fiyati, $product_id]);
            $success_message = 'Ürün başarıyla güncellendi.';
        }
        
        if ($success_message) {
            header('Location: products.php?message=success');
            exit();
        }
        
    } catch (Exception $e) {
        $error_message = $e->getMessage();
    }
}

if ($action === 'delete' && $product_id && $_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        validateCSRF();
        
        Authorization::requirePageAccess('products', 'silme');
        
        $sql = "SELECT COUNT(*) as count FROM satis_detay WHERE urun_id = ?";
        $result = $database->fetch($sql, [$product_id]);
        
        if ($result['count'] > 0) {
            $error_message = 'Bu ürünün satış kaydı bulunduğu için silinemez.';
        } else {
            $sql = "DELETE FROM urunler WHERE urun_id = ?";
            $database->execute($sql, [$product_id]);
            header('Location: products.php?message=deleted');
            exit();
        }
    } catch (Exception $e) {
        $error_message = 'Ürün silinirken hata oluştu: ' . $e->getMessage();
    }
}

if (isset($_GET['message'])) {
    switch ($_GET['message']) {
        case 'success':
            $success_message = 'İşlem başarıyla tamamlandı.';
            break;
        case 'deleted':
            $success_message = 'Ürün başarıyla silindi.';
            break;
    }
}

$search = Security::sanitizeInput($_GET['search'] ?? '');
$filter_stock = $_GET['filter_stock'] ?? '';

$where_conditions = [];
$params = [];

if (!empty($search)) {
    $where_conditions[] = "(urun_adi LIKE ? OR barkod LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($filter_stock === 'low') {
    $where_conditions[] = "stok_miktari < 10";
} elseif ($filter_stock === 'zero') {
    $where_conditions[] = "stok_miktari = 0";
}

$where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';

$sql = "SELECT * FROM urunler $where_clause ORDER BY urun_adi ASC";
$products = $database->fetchAll($sql, $params);

$edit_product = null;
if ($action === 'edit' && $product_id) {
    $sql = "SELECT * FROM urunler WHERE urun_id = ?";
    $edit_product = $database->fetch($sql, [$product_id]);
    if (!$edit_product) {
        $error_message = 'Ürün bulunamadı.';
        $action = 'list';
    }
}
?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ürün Yönetimi - Depo Stok Takip Sistemi</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
        }
        .btn {
            border-radius: 10px;
        }
        .form-control {
            border-radius: 10px;
            border: 2px solid #e9ecef;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .table th {
            background: #f8f9fa;
            border: none;
            font-weight: 600;
        }
        .badge {
            font-size: 0.75rem;
        }

    </style>
</head>
<body>
    <?php include '../includes/header.php'; ?>
    <?php include '../includes/sidebar.php'; ?>

    
    <div class="main-content">
        <div class="container-fluid">
            <div class="d-flex justify-content-between align-items-center mb-4">
                <h2>
                    <i class="fas fa-boxes me-2"></i>
                    Ürün Yönetimi
                    <?php if ($action === 'add'): ?>
                        - Yeni Ürün Ekle
                    <?php elseif ($action === 'edit'): ?>
                        - Ürün Düzenle
                    <?php endif; ?>
                </h2>
                <a href="dashboard.php" class="btn btn-outline-secondary">
                    <i class="fas fa-arrow-left me-1"></i>
                    Dashboard
                </a>
            </div>
            <?php if ($error_message): ?>
                <div class="alert alert-danger" role="alert">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <?php echo Security::escapeOutput($error_message); ?>
                </div>
            <?php endif; ?>

            <?php if ($success_message): ?>
                <div class="alert alert-success" role="alert">
                    <i class="fas fa-check-circle me-2"></i>
                    <?php echo Security::escapeOutput($success_message); ?>
                </div>
            <?php endif; ?>

            <?php if ($action === 'list'): ?>
                
                <div class="row mb-4">
                    <div class="col-md-6">
                        <?php if (Authorization::getUserPagePermissions('products')['yazma']): ?>
                        <a href="?action=add" class="btn btn-primary">
                            <i class="fas fa-plus me-2"></i>
                            Yeni Ürün Ekle
                        </a>
                        <?php endif; ?>
                    </div>
                    <div class="col-md-6">
                        <form method="GET" class="d-flex">
                            <input type="text" 
                                   class="form-control me-2" 
                                   name="search" 
                                   placeholder="Ürün adı veya barkod ara..."
                                   value="<?php echo Security::escapeOutput($search); ?>">
                            <select name="filter_stock" class="form-select me-2" style="width: auto;">
                                <option value="">Tüm Stoklar</option>
                                <option value="low" <?php echo $filter_stock === 'low' ? 'selected' : ''; ?>>Düşük Stok (&lt;10)</option>
                                <option value="zero" <?php echo $filter_stock === 'zero' ? 'selected' : ''; ?>>Stok Yok</option>
                            </select>
                            <button type="submit" class="btn btn-outline-primary">
                                <i class="fas fa-search"></i>
                            </button>
                        </form>
                    </div>
                </div>

                <div class="card">
                    <div class="card-body">
                        <?php if (empty($products)): ?>
                            <div class="text-center py-5">
                                <i class="fas fa-boxes fa-3x text-muted mb-3"></i>
                                <h5 class="text-muted">Ürün bulunamadı</h5>
                                <p class="text-muted">
                                    <?php if (!empty($search) || !empty($filter_stock)): ?>
                                        Arama kriterlerinize uygun ürün bulunamadı.
                                    <?php else: ?>
                                        Henüz hiç ürün eklenmemiş.
                                    <?php endif; ?>
                                </p>
                                <a href="?action=add" class="btn btn-primary">
                                    <i class="fas fa-plus me-2"></i>
                                    İlk Ürünü Ekle
                                </a>
                            </div>
                        <?php else: ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Ürün Adı</th>
                                            <th>Barkod</th>
                                            <th>Stok</th>
                                            <th>Alış Fiyatı</th>
                                            <th>Satış Fiyatı</th>
                                            <th>Kar Marjı</th>
                                            <th>İşlemler</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($products as $product): ?>
                                            <?php 
                                            $kar_marji = $product['alis_fiyati'] > 0 ? 
                                                (($product['satis_fiyati'] - $product['alis_fiyati']) / $product['alis_fiyati']) * 100 : 0;
                                            ?>
                                            <tr>
                                                <td>
                                                    <strong><?php echo Security::escapeOutput($product['urun_adi']); ?></strong>
                                                </td>
                                                <td>
                                                    <?php if ($product['barkod']): ?>
                                                        <code><?php echo Security::escapeOutput($product['barkod']); ?></code>
                                                    <?php else: ?>
                                                        <span class="text-muted">-</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <?php if ($product['stok_miktari'] == 0): ?>
                                                        <span class="badge bg-danger"><?php echo $product['stok_miktari']; ?></span>
                                                    <?php elseif ($product['stok_miktari'] < 10): ?>
                                                        <span class="badge bg-warning"><?php echo $product['stok_miktari']; ?></span>
                                                    <?php else: ?>
                                                        <span class="badge bg-success"><?php echo $product['stok_miktari']; ?></span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>₺<?php echo number_format($product['alis_fiyati'], 2); ?></td>
                                                <td>₺<?php echo number_format($product['satis_fiyati'], 2); ?></td>
                                                <td>
                                                    <?php if ($kar_marji > 0): ?>
                                                        <span class="text-success">+%<?php echo number_format($kar_marji, 1); ?></span>
                                                    <?php elseif ($kar_marji < 0): ?>
                                                        <span class="text-danger">%<?php echo number_format($kar_marji, 1); ?></span>
                                                    <?php else: ?>
                                                        <span class="text-muted">%0</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <div class="btn-group btn-group-sm">
                                                        <?php if (Authorization::getUserPagePermissions('products')['yazma']): ?>
                                                        <a href="?action=edit&id=<?php echo $product['urun_id']; ?>" 
                                                           class="btn btn-outline-primary">
                                                            <i class="fas fa-edit"></i>
                                                        </a>
                                                        <?php endif; ?>
                                                        <?php if (Authorization::getUserPagePermissions('products')['silme']): ?>
                                                        <button type="button" 
                                                                class="btn btn-outline-danger"
                                                                onclick="confirmDelete(<?php echo $product['urun_id']; ?>, '<?php echo Security::escapeOutput($product['urun_adi']); ?>')">
                                                            <i class="fas fa-trash"></i>
                                                        </button>
                                                        <?php endif; ?>
                                                    </div>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

            <?php elseif ($action === 'add' || $action === 'edit'): ?>
                
                <div class="row justify-content-center">
                    <div class="col-md-8">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">
                                    <i class="fas fa-<?php echo $action === 'add' ? 'plus' : 'edit'; ?> me-2"></i>
                                    <?php echo $action === 'add' ? 'Yeni Ürün Ekle' : 'Ürün Düzenle'; ?>
                                </h5>
                            </div>
                            <div class="card-body">
                                <form method="POST" novalidate>
                                    <?php echo Security::getCSRFTokenInput(); ?>
                                    
                                    <div class="row">
                                        <div class="col-md-8 mb-3">
                                            <label for="urun_adi" class="form-label">Ürün Adı *</label>
                                            <input type="text" 
                                                   class="form-control" 
                                                   id="urun_adi" 
                                                   name="urun_adi" 
                                                   value="<?php echo Security::escapeOutput($edit_product['urun_adi'] ?? ''); ?>"
                                                   required>
                                        </div>
                                        <div class="col-md-4 mb-3">
                                            <label for="barkod" class="form-label">Barkod</label>
                                            <input type="text" 
                                                   class="form-control" 
                                                   id="barkod" 
                                                   name="barkod" 
                                                   value="<?php echo Security::escapeOutput($edit_product['barkod'] ?? ''); ?>">
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-4 mb-3">
                                            <label for="stok_miktari" class="form-label">Stok Miktarı *</label>
                                            <input type="number" 
                                                   class="form-control" 
                                                   id="stok_miktari" 
                                                   name="stok_miktari" 
                                                   value="<?php echo $edit_product['stok_miktari'] ?? 0; ?>"
                                                   min="0" 
                                                   required>
                                        </div>
                                        <div class="col-md-4 mb-3">
                                            <label for="alis_fiyati" class="form-label">Alış Fiyatı (₺) *</label>
                                            <input type="number" 
                                                   class="form-control" 
                                                   id="alis_fiyati" 
                                                   name="alis_fiyati" 
                                                   value="<?php echo $edit_product['alis_fiyati'] ?? ''; ?>"
                                                   step="0.01" 
                                                   min="0" 
                                                   required>
                                        </div>
                                        <div class="col-md-4 mb-3">
                                            <label for="satis_fiyati" class="form-label">Satış Fiyatı (₺) *</label>
                                            <input type="number" 
                                                   class="form-control" 
                                                   id="satis_fiyati" 
                                                   name="satis_fiyati" 
                                                   value="<?php echo $edit_product['satis_fiyati'] ?? ''; ?>"
                                                   step="0.01" 
                                                   min="0" 
                                                   required>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="alert alert-info">
                                                <small>
                                                    <i class="fas fa-info-circle me-1"></i>
                                                    <strong>Kar Marjı:</strong> <span id="kar_marji">%0</span>
                                                </small>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="d-flex justify-content-between">
                                        <a href="products.php" class="btn btn-secondary">
                                            <i class="fas fa-arrow-left me-2"></i>
                                            Geri Dön
                                        </a>
                                        <button type="submit" class="btn btn-primary">
                                            <i class="fas fa-save me-2"></i>
                                            <?php echo $action === 'add' ? 'Ürün Ekle' : 'Güncelle'; ?>
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function calculateMargin() {
            const alisFiyati = parseFloat(document.getElementById('alis_fiyati').value) || 0;
            const satisFiyati = parseFloat(document.getElementById('satis_fiyati').value) || 0;
            
            if (alisFiyati > 0) {
                const margin = ((satisFiyati - alisFiyati) / alisFiyati) * 100;
                document.getElementById('kar_marji').textContent = 
                    (margin >= 0 ? '+' : '') + margin.toFixed(1) + '%';
                document.getElementById('kar_marji').className = 
                    margin >= 0 ? 'text-success' : 'text-danger';
            } else {
                document.getElementById('kar_marji').textContent = '%0';
                document.getElementById('kar_marji').className = 'text-muted';
            }
        }

        document.getElementById('alis_fiyati')?.addEventListener('input', calculateMargin);
        document.getElementById('satis_fiyati')?.addEventListener('input', calculateMargin);

        calculateMargin();

        function confirmDelete(id, name) {
            if (confirm(`"${name}" adlı ürünü silmek istediğinizden emin misiniz?\n\nBu işlem geri alınamaz.`)) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = `?action=delete&id=${id}`;
                
                const csrfInput = document.createElement('input');
                csrfInput.type = 'hidden';
                csrfInput.name = 'csrf_token';
                csrfInput.value = '<?php echo Security::generateCSRFToken(); ?>';
                form.appendChild(csrfInput);
                
                document.body.appendChild(form);
                form.submit();
            }
        }

        document.querySelector('form')?.addEventListener('submit', function(e) {
            const urunAdi = document.getElementById('urun_adi').value.trim();
            const alisFiyati = parseFloat(document.getElementById('alis_fiyati').value);
            const satisFiyati = parseFloat(document.getElementById('satis_fiyati').value);

            if (!urunAdi) {
                e.preventDefault();
                alert('Ürün adı gereklidir.');
                return false;
            }

            if (alisFiyati < 0 || satisFiyati < 0) {
                e.preventDefault();
                alert('Fiyatlar negatif olamaz.');
                return false;
            }
        });
    </script>
</body>
</html>