<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
require_once '../config/config.php';
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../includes/performance_monitor.php';

requireLogin();

Authorization::requirePageAccess('performance_monitor', 'okuma');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!Security::validateCSRFToken($_POST['csrf_token'] ?? '')) {
        die('CSRF token hatası!');
    }
}

$monitor = new PerformanceMonitor($database);

$action = $_GET['action'] ?? 'dashboard';
$message = '';
$error = '';

try {
    switch ($action) {
        case 'cleanup':
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                Authorization::requirePageAccess('performance_monitor', 'yazma');
                
                $days = (int)($_POST['days'] ?? 7);
                $deleted = $monitor->cleanupOldLogs($days);
                $message = "$deleted adet eski performans kaydı temizlendi.";
            }
            break;
            
        case 'export':
            $report = $monitor->generateReport();
            header('Content-Type: application/json');
            header('Content-Disposition: attachment; filename="performance_report_' . date('Y-m-d_H-i-s') . '.json"');
            echo json_encode($report, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            exit;
            break;
    }
    
    $performance_data = $monitor->generateReport();
    
} catch (Exception $e) {
    $error = "Performans verileri alınırken hata oluştu: " . $e->getMessage();
    error_log($error);
}

$monitor->logOperation('performance_monitor_view', 'performance_monitor');

$csrf_token = Security::generateCSRFToken();

include '../includes/header.php';
?>

<style>
    .performance-card {
        transition: transform 0.2s;
    }
    .performance-card:hover {
        transform: translateY(-2px);
    }
    .metric-value {
        font-size: 2rem;
        font-weight: bold;
    }
    .recommendation-warning {
        border-left: 4px solid #ffc107;
    }
    .recommendation-error {
        border-left: 4px solid #dc3545;
    }
    .recommendation-info {
        border-left: 4px solid #0dcaf0;
    }
    .table-size-bar {
        height: 20px;
        background: linear-gradient(90deg, #28a745, #ffc107, #dc3545);
        border-radius: 10px;
    }
</style>

<?php include '../includes/sidebar.php'; ?>

<div class="main-content">
    <div class="container-fluid">
        <?php if ($message): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <i class="fas fa-check-circle me-2"></i><?= htmlspecialchars($message) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if ($error): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i><?= htmlspecialchars($error) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card performance-card h-100">
                    <div class="card-body text-center">
                        <i class="fas fa-memory fa-2x text-primary mb-2"></i>
                        <h6 class="card-title">Bellek Kullanımı</h6>
                        <div class="metric-value text-primary">
                            <?= $performance_data['system_stats']['memory_usage']['current'] ?? 0 ?> MB
                        </div>
                        <small class="text-muted">
                            Peak: <?= $performance_data['system_stats']['memory_usage']['peak'] ?? 0 ?> MB
                        </small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card performance-card h-100">
                    <div class="card-body text-center">
                        <i class="fas fa-clock fa-2x text-warning mb-2"></i>
                        <h6 class="card-title">Yükleme Süresi</h6>
                        <div class="metric-value text-warning">
                            <?= $performance_data['system_stats']['execution_time'] ?? 0 ?>s
                        </div>
                        <small class="text-muted">Sayfa yükleme süresi</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card performance-card h-100">
                    <div class="card-body text-center">
                        <i class="fas fa-database fa-2x text-success mb-2"></i>
                        <h6 class="card-title">Aktif Bağlantı</h6>
                        <div class="metric-value text-success">
                            <?= $performance_data['database_stats']['active_connections'] ?? 0 ?>
                        </div>
                        <small class="text-muted">Veritabanı bağlantıları</small>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card performance-card h-100">
                    <div class="card-body text-center">
                        <i class="fas fa-hdd fa-2x text-info mb-2"></i>
                        <h6 class="card-title">Disk Alanı</h6>
                        <div class="metric-value text-info">
                            <?= $performance_data['system_stats']['disk_free'] ?? 'N/A' ?>
                        </div>
                        <small class="text-muted">Boş disk alanı</small>
                    </div>
                </div>
            </div>
        </div>

        
        <?php if (!empty($performance_data['recommendations'])): ?>
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-lightbulb me-2"></i>Performans Önerileri
                </h5>
            </div>
            <div class="card-body">
                <?php foreach ($performance_data['recommendations'] as $rec): ?>
                <div class="alert recommendation-<?= $rec['type'] ?> mb-2">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <h6 class="alert-heading mb-1"><?= htmlspecialchars($rec['title']) ?></h6>
                            <p class="mb-1"><?= htmlspecialchars($rec['message']) ?></p>
                            <small><strong>Önerilen Aksiyon:</strong> <?= htmlspecialchars($rec['action']) ?></small>
                        </div>
                        <i class="fas fa-<?= $rec['type'] === 'error' ? 'exclamation-triangle' : ($rec['type'] === 'warning' ? 'exclamation-circle' : 'info-circle') ?>"></i>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
        </div>
        <?php endif; ?>

        <div class="row">
            
            <div class="col-md-6 mb-4">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="fas fa-table me-2"></i>Tablo Boyutları
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($performance_data['database_stats']['table_sizes'])): ?>
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Tablo</th>
                                        <th>Boyut (MB)</th>
                                        <th>Kayıt Sayısı</th>
                                        <th>Görsel</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $max_size = max(array_column($performance_data['database_stats']['table_sizes'], 'size_mb'));
                                    foreach ($performance_data['database_stats']['table_sizes'] as $table): 
                                        $width = $max_size > 0 ? ($table['size_mb'] / $max_size) * 100 : 0;
                                    ?>
                                    <tr>
                                        <td><?= htmlspecialchars($table['table_name']) ?></td>
                                        <td><?= number_format($table['size_mb'], 2) ?></td>
                                        <td><?= number_format($table['table_rows']) ?></td>
                                        <td>
                                            <div class="progress" style="height: 15px;">
                                                <div class="progress-bar" style="width: <?= $width ?>%"></div>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <p class="text-muted">Tablo boyut bilgisi alınamadı.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            
            <div class="col-md-6 mb-4">
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="fas fa-turtle me-2"></i>Yavaş İşlemler (Son 24 Saat)
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if (!empty($performance_data['slow_queries'])): ?>
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>İşlem</th>
                                        <th>Tablo</th>
                                        <th>Süre (s)</th>
                                        <th>Bellek (MB)</th>
                                        <th>Tarih</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($performance_data['slow_queries'] as $query): ?>
                                    <tr>
                                        <td><?= htmlspecialchars($query['operation_type']) ?></td>
                                        <td><?= htmlspecialchars($query['table_name'] ?? '-') ?></td>
                                        <td>
                                            <span class="badge bg-<?= $query['execution_time'] > 5 ? 'danger' : ($query['execution_time'] > 2 ? 'warning' : 'success') ?>">
                                                <?= number_format($query['execution_time'], 2) ?>
                                            </span>
                                        </td>
                                        <td><?= number_format($query['memory_usage'], 2) ?></td>
                                        <td><?= date('H:i:s', strtotime($query['created_at'])) ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php else: ?>
                        <p class="text-muted">Son 24 saatte yavaş işlem kaydı bulunamadı.</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-server me-2"></i>Sistem Bilgileri
                </h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h6>PHP Konfigürasyonu</h6>
                        <ul class="list-unstyled">
                            <li><strong>PHP Sürümü:</strong> <?= $performance_data['system_stats']['php_config']['version'] ?? 'N/A' ?></li>
                            <li><strong>Bellek Limiti:</strong> <?= $performance_data['system_stats']['memory_usage']['limit'] ?? 'N/A' ?></li>
                            <li><strong>Max Execution Time:</strong> <?= $performance_data['system_stats']['php_config']['max_execution_time'] ?? 'N/A' ?>s</li>
                            <li><strong>Upload Max Size:</strong> <?= $performance_data['system_stats']['php_config']['upload_max_filesize'] ?? 'N/A' ?></li>
                        </ul>
                    </div>
                    <div class="col-md-6">
                        <h6>Veritabanı Durumu</h6>
                        <ul class="list-unstyled">
                            <li><strong>Slow Query Log:</strong> 
                                <span class="badge bg-<?= ($performance_data['database_stats']['slow_query_log'] ?? 'OFF') === 'ON' ? 'success' : 'warning' ?>">
                                    <?= $performance_data['database_stats']['slow_query_log'] ?? 'OFF' ?>
                                </span>
                            </li>
                            <li><strong>Aktif Bağlantılar:</strong> <?= $performance_data['database_stats']['active_connections'] ?? 0 ?></li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>

        
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">
                    <i class="fas fa-tools me-2"></i>Performans İşlemleri
                </h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <form method="post" action="?action=cleanup" class="mb-3">
                            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                            <div class="input-group">
                                <input type="number" name="days" class="form-control" value="7" min="1" max="30">
                                <span class="input-group-text">gün önceki</span>
                                <button type="submit" class="btn btn-warning">
                                    <i class="fas fa-broom me-1"></i>Logları Temizle
                                </button>
                            </div>
                            <small class="text-muted">Eski performans loglarını temizler</small>
                        </form>
                    </div>
                    <div class="col-md-6">
                        <a href="?action=export" class="btn btn-info">
                            <i class="fas fa-download me-1"></i>Raporu İndir (JSON)
                        </a>
                        <button type="button" class="btn btn-secondary" onclick="location.reload()">
                            <i class="fas fa-sync-alt me-1"></i>Yenile
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        setTimeout(function() {
            location.reload();
        }, 30000);
    });
</script>

<?php include '../includes/footer.php'; ?>