<?php
session_start();
require_once '../config/config.php';
require_once '../includes/security.php';
require_once '../includes/authorization.php';
require_once '../includes/report_manager.php';

if (!Security::isLoggedIn() || !Security::checkSessionTimeout()) {
    header('Location: login.php');
    exit();
}

Authorization::requirePageAccess('advanced_reports', 'okuma');

$reportManager = new ReportManager();
$message = '';
$error = '';
$reportData = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        Security::validateCSRFToken($_POST['csrf_token'] ?? '');
    } catch (Exception $e) {
        $error = 'Güvenlik hatası! Lütfen sayfayı yenileyin.';
    }
    
    if (empty($error)) {
        $action = $_POST['action'] ?? '';
        
        try {
            switch ($action) {
                case 'generate_report':
                    $reportId = (int)($_POST['report_id'] ?? 0);
                    $parameters = [];
                    
                    foreach ($_POST as $key => $value) {
                        if (strpos($key, 'param_') === 0) {
                            $paramName = substr($key, 6);
                            $parameters[$paramName] = Security::sanitizeInput($value);
                        }
                    }
                    
                    $reportData = $reportManager->generateReport($reportId, $parameters, $_SESSION['user_id']);
                    
                    if ($reportData['success']) {
                        $message = 'Rapor başarıyla oluşturuldu.';
                    } else {
                        $error = 'Rapor oluşturulurken hata: ' . $reportData['error'];
                    }
                    break;
                    
                case 'export_excel':
                    if (isset($_SESSION['last_report_data'])) {
                        $export = $reportManager->exportToExcel($_SESSION['last_report_data']);
                        if ($export['success']) {
                            header('Content-Type: ' . $export['mime_type']);
                            header('Content-Disposition: attachment; filename="' . $export['filename'] . '"');
                            echo $export['content'];
                            exit;
                        } else {
                            $error = 'Excel dışa aktarma hatası: ' . $export['error'];
                        }
                    }
                    break;
                    
                case 'export_pdf':
                    if (isset($_SESSION['last_report_data'])) {
                        $export = $reportManager->exportToPDF($_SESSION['last_report_data']);
                        if ($export['success']) {
                            header('Content-Type: ' . $export['mime_type']);
                            header('Content-Disposition: attachment; filename="' . $export['filename'] . '"');
                            echo $export['content'];
                            exit;
                        } else {
                            $error = 'PDF dışa aktarma hatası: ' . $export['error'];
                        }
                    }
                    break;
            }
        } catch (Exception $e) {
            $error = 'İşlem hatası: ' . $e->getMessage();
        }
    }
}

if ($reportData && $reportData['success']) {
    $_SESSION['last_report_data'] = $reportData;
}

$reportTemplates = $reportManager->getReportTemplates();
$widgets = $reportManager->getAllWidgets();
$dashboardStats = $reportManager->getDashboardStats();
$reportHistory = $reportManager->getReportHistory($_SESSION['user_id'], 10);

$csrf_token = Security::generateCSRFToken();

include '../includes/header.php';
?>


<?php include '../includes/sidebar.php'; ?>


<div class="main-content">
    <style>
        .widget-card {
            transition: transform 0.2s;
            cursor: pointer;
        }
        .widget-card:hover {
            transform: translateY(-5px);
        }
        .report-table {
            max-height: 500px;
            overflow-y: auto;
        }
        .chart-container {
            position: relative;
            height: 300px;
        }
    </style>
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>Gelişmiş Raporlama Sistemi</h2>
                    <div>
                        <span class="badge bg-primary">Kullanıcı: <?= htmlspecialchars($_SESSION['username']) ?></span>
                    </div>
                </div>

                <?php if ($message): ?>
                    <div class="alert alert-success alert-dismissible fade show">
                        <?= htmlspecialchars($message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show">
                        <?= htmlspecialchars($error) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                
                <div class="tab-content">
                    
                    <div class="tab-pane fade show active" id="dashboard-tab">
                        <div class="row mb-4">
                            
                            <div class="col-md-3 mb-3">
                                <div class="card widget-card bg-primary text-white">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between">
                                            <div>
                                                <h6 class="card-title">Toplam Ürün</h6>
                                                <h3><?= number_format($dashboardStats['stok']['toplam_urun'] ?? 0) ?></h3>
                                            </div>
                                            <div class="align-self-center">
                                                <i class="fas fa-boxes fa-2x"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-3 mb-3">
                                <div class="card widget-card bg-danger text-white">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between">
                                            <div>
                                                <h6 class="card-title">Kritik Stok</h6>
                                                <h3><?= number_format($dashboardStats['stok']['kritik_stok'] ?? 0) ?></h3>
                                            </div>
                                            <div class="align-self-center">
                                                <i class="fas fa-exclamation-triangle fa-2x"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-3 mb-3">
                                <div class="card widget-card bg-success text-white">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between">
                                            <div>
                                                <h6 class="card-title">Stok Değeri</h6>
                                                <h3>₺<?= number_format($dashboardStats['stok']['toplam_stok_degeri'] ?? 0, 2) ?></h3>
                                            </div>
                                            <div class="align-self-center">
                                                <i class="fas fa-dollar-sign fa-2x"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-3 mb-3">
                                <div class="card widget-card bg-info text-white">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between">
                                            <div>
                                                <h6 class="card-title">Aylık Rapor</h6>
                                                <h3><?= number_format($dashboardStats['raporlar']['toplam'] ?? 0) ?></h3>
                                            </div>
                                            <div class="align-self-center">
                                                <i class="fas fa-chart-bar fa-2x"></i>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-header">
                                        <h5>Stok Dağılımı</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="chart-container">
                                            <canvas id="stockChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-header">
                                        <h5>Nakit ve Borç Durumu</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="chart-container">
                                            <canvas id="cashDebtChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row mt-4">
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-header">
                                        <h5>Müşteri Alacakları</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="chart-container">
                                            <canvas id="customerReceivablesChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="card">
                                    <div class="card-header">
                                        <h5>Tedarikçi Borçları</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="chart-container">
                                            <canvas id="supplierDebtsChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Stok Giriş/Çıkış Değer Karşılaştırması -->
                        <div class="row mt-4">
                            <div class="col-12">
                                <div class="card">
                                    <div class="card-header d-flex justify-content-between align-items-center">
                                        <h5>Stok Giriş/Çıkış Değer Karşılaştırması</h5>
                                        <div class="btn-group" role="group">
                                            <button type="button" class="btn btn-outline-primary btn-sm" onclick="updateStockComparison('monthly')">Aylık</button>
                                            <button type="button" class="btn btn-outline-primary btn-sm active" onclick="updateStockComparison('yearly')">Yıllık</button>
                                        </div>
                                    </div>
                                    <div class="card-body">
                                        <div class="row mb-3">
                                            <div class="col-md-3">
                                                <div class="text-center">
                                                    <h4 class="text-success">₺<?= number_format($dashboardStats['stok_hareket']['toplam_giris'] ?? 0, 2) ?></h4>
                                                    <small class="text-muted">Toplam Stok Girişi</small>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="text-center">
                                                    <h4 class="text-danger">₺<?= number_format($dashboardStats['stok_hareket']['toplam_cikis'] ?? 0, 2) ?></h4>
                                                    <small class="text-muted">Toplam Satış</small>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="text-center">
                                                    <?php 
                                                    $kar_zarar = ($dashboardStats['stok_hareket']['toplam_cikis'] ?? 0) - ($dashboardStats['stok_hareket']['toplam_giris'] ?? 0);
                                                    $kar_zarar_class = $kar_zarar >= 0 ? 'text-success' : 'text-danger';
                                                    ?>
                                                    <h4 class="<?= $kar_zarar_class ?>">₺<?= number_format($kar_zarar, 2) ?></h4>
                                                    <small class="text-muted"><?= $kar_zarar >= 0 ? 'Brüt Kar' : 'Zarar' ?></small>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="text-center">
                                                    <?php 
                                                    $kar_orani = ($dashboardStats['stok_hareket']['toplam_giris'] ?? 0) > 0 
                                                        ? (($kar_zarar / ($dashboardStats['stok_hareket']['toplam_giris'] ?? 1)) * 100) 
                                                        : 0;
                                                    $kar_orani_class = $kar_orani >= 0 ? 'text-success' : 'text-danger';
                                                    ?>
                                                    <h4 class="<?= $kar_orani_class ?>">%<?= number_format($kar_orani, 1) ?></h4>
                                                    <small class="text-muted">Kar Oranı</small>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="chart-container">
                                            <canvas id="stockComparisonChart"></canvas>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    
                    <div class="tab-pane fade" id="reports-tab">
                        <div class="row">
                            <div class="col-md-4">
                                <div class="card">
                                    <div class="card-header">
                                        <h5>Rapor Oluştur</h5>
                                    </div>
                                    <div class="card-body">
                                        <form method="POST">
                                            <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                            <input type="hidden" name="action" value="generate_report">
                                            
                                            <div class="mb-3">
                                                <label class="form-label">Rapor Şablonu</label>
                                                <select name="report_id" class="form-select" required onchange="updateParameters(this.value)">
                                                    <option value="">Seçiniz...</option>
                                                    <?php foreach ($reportTemplates as $template): ?>
                                                        <option value="<?= $template['rapor_id'] ?>" 
                                                                data-params='<?= htmlspecialchars($template['rapor_parametreleri']) ?>'>
                                                            <?= htmlspecialchars($template['rapor_adi']) ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                            
                                            <div id="parameters-container">
                                                
                                            </div>
                                            
                                            <button type="submit" class="btn btn-primary">
                                                <i class="fas fa-play me-2"></i>Rapor Oluştur
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-8">
                                <?php if ($reportData && $reportData['success']): ?>
                                    <div class="card">
                                        <div class="card-header d-flex justify-content-between align-items-center">
                                            <h5><?= htmlspecialchars($reportData['template']['rapor_adi']) ?></h5>
                                            <div>
                                                <form method="POST" class="d-inline">
                                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                                    <input type="hidden" name="action" value="export_excel">
                                                    <button type="submit" class="btn btn-success btn-sm">
                                                        <i class="fas fa-file-excel me-1"></i>Excel
                                                    </button>
                                                </form>
                                                <form method="POST" class="d-inline">
                                                    <input type="hidden" name="csrf_token" value="<?= $csrf_token ?>">
                                                    <input type="hidden" name="action" value="export_pdf">
                                                    <button type="submit" class="btn btn-danger btn-sm">
                                                        <i class="fas fa-file-pdf me-1"></i>HTML
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                        <div class="card-body">
                                            <div class="report-table">
                                                <?php if (!empty($reportData['data'])): ?>
                                                    <table class="table table-striped table-hover">
                                                        <thead class="table-dark">
                                                            <tr>
                                                                <?php foreach (array_keys($reportData['data'][0]) as $header): ?>
                                                                    <th><?= htmlspecialchars($header) ?></th>
                                                                <?php endforeach; ?>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                            <?php foreach ($reportData['data'] as $row): ?>
                                                                <tr>
                                                                    <?php foreach ($row as $cell): ?>
                                                                        <td><?= htmlspecialchars($cell) ?></td>
                                                                    <?php endforeach; ?>
                                                                </tr>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                <?php else: ?>
                                                    <div class="alert alert-info">
                                                        <i class="fas fa-info-circle me-2"></i>
                                                        Bu rapor için veri bulunamadı.
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>
                                <?php else: ?>
                                    <div class="card">
                                        <div class="card-body text-center text-muted">
                                            <i class="fas fa-chart-bar fa-3x mb-3"></i>
                                            <h5>Rapor Seçin</h5>
                                            <p>Sol taraftan bir rapor şablonu seçin ve parametreleri girerek rapor oluşturun.</p>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    
                    <div class="tab-pane fade" id="history-tab">
                        <div class="card">
                            <div class="card-header">
                                <h5>Rapor Geçmişi</h5>
                            </div>
                            <div class="card-body">
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead>
                                            <tr>
                                                <th>Rapor Adı</th>
                                                <th>Oluşturulma Tarihi</th>
                                                <th>Durum</th>
                                                <th>Parametreler</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($reportHistory as $history): ?>
                                                <tr>
                                                    <td><?= htmlspecialchars($history['rapor_adi']) ?></td>
                                                    <td><?= date('d.m.Y H:i', strtotime($history['olusturulma_tarihi'])) ?></td>
                                                    <td>
                                                        <?php if ($history['rapor_durumu'] === 'tamamlandi'): ?>
                                                            <span class="badge bg-success">Tamamlandı</span>
                                                        <?php elseif ($history['rapor_durumu'] === 'hata'): ?>
                                                            <span class="badge bg-danger">Hata</span>
                                                        <?php else: ?>
                                                            <span class="badge bg-warning">Oluşturuluyor</span>
                                                        <?php endif; ?>
                                                    </td>
                                                    <td>
                                                        <?php 
                                                        $params = json_decode($history['rapor_parametreleri'], true);
                                                        if ($params) {
                                                            echo '<small>' . htmlspecialchars(implode(', ', array_map(function($k, $v) {
                                                                return "$k: $v";
                                                            }, array_keys($params), $params))) . '</small>';
                                                        }
                                                        ?>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
    <script>
        function updateParameters(reportId) {
            const select = document.querySelector('select[name="report_id"]');
            const option = select.querySelector(`option[value="${reportId}"]`);
            const container = document.getElementById('parameters-container');
            
            container.innerHTML = '';
            
            if (option && option.dataset.params) {
                try {
                    const params = JSON.parse(option.dataset.params);
                    
                    for (const [key, defaultValue] of Object.entries(params)) {
                        const div = document.createElement('div');
                        div.className = 'mb-3';
                        
                        let inputType = 'text';
                        if (key.includes('tarih')) {
                            inputType = 'date';
                        } else if (key.includes('miktar') || key.includes('fiyat')) {
                            inputType = 'number';
                        }
                        
                        div.innerHTML = `
                            <label class="form-label">${key.replace(/_/g, ' ').toUpperCase()}</label>
                            <input type="${inputType}" name="param_${key}" class="form-control" 
                                   value="${defaultValue}" placeholder="${defaultValue}">
                        `;
                        
                        container.appendChild(div);
                    }
                } catch (e) {
                    console.error('Parametre parse hatası:', e);
                }
            }
        }

        const stockData = {
            labels: ['Normal Stok', 'Düşük Stok', 'Kritik Stok'],
            datasets: [{
                data: [
                    <?= $dashboardStats['stok']['normal_stok'] ?? 0 ?>,
                    <?= $dashboardStats['stok']['dusuk_stok'] ?? 0 ?>,
                    <?= $dashboardStats['stok']['kritik_stok'] ?? 0 ?>
                ],
                backgroundColor: ['#28a745', '#ffc107', '#dc3545']
            }]
        };

        new Chart(document.getElementById('stockChart'), {
            type: 'doughnut',
            data: stockData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Nakit ve Borç Durumu Grafiği
        const cashDebtData = {
            labels: ['Müşteri Alacakları', 'Tedarikçi Borçları', 'Net Bakiye'],
            datasets: [{
                data: [
                    <?= $dashboardStats['cari']['musteri_alacak'] ?? 0 ?>,
                    <?= $dashboardStats['cari']['tedarikci_borc'] ?? 0 ?>,
                    <?= ($dashboardStats['cari']['musteri_alacak'] ?? 0) - ($dashboardStats['cari']['tedarikci_borc'] ?? 0) ?>
                ],
                backgroundColor: ['#28a745', '#dc3545', '#007bff']
            }]
        };

        new Chart(document.getElementById('cashDebtChart'), {
            type: 'doughnut',
            data: cashDebtData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Müşteri Alacakları Grafiği
        const customerReceivablesData = {
            labels: ['Alacaklı Müşteri', 'Borçlu Müşteri', 'Sıfır Bakiye'],
            datasets: [{
                data: [
                    <?= $dashboardStats['cari']['alacakli_musteri'] ?? 0 ?>,
                    <?= $dashboardStats['cari']['borclu_musteri'] ?? 0 ?>,
                    <?= ($dashboardStats['cari']['toplam_musteri'] ?? 0) - ($dashboardStats['cari']['alacakli_musteri'] ?? 0) - ($dashboardStats['cari']['borclu_musteri'] ?? 0) ?>
                ],
                backgroundColor: ['#28a745', '#ffc107', '#6c757d']
            }]
        };

        new Chart(document.getElementById('customerReceivablesChart'), {
            type: 'bar',
            data: customerReceivablesData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Tedarikçi Borçları Grafiği
        const supplierDebtsData = {
            labels: ['Borcumuz Olan Tedarikçi', 'Alacağımız Olan Tedarikçi', 'Sıfır Bakiye'],
            datasets: [{
                data: [
                    <?= $dashboardStats['cari']['borclu_tedarikci'] ?? 0 ?>,
                    <?= $dashboardStats['cari']['alacakli_tedarikci'] ?? 0 ?>,
                    <?= ($dashboardStats['cari']['toplam_tedarikci'] ?? 0) - ($dashboardStats['cari']['borclu_tedarikci'] ?? 0) - ($dashboardStats['cari']['alacakli_tedarikci'] ?? 0) ?>
                ],
                backgroundColor: ['#dc3545', '#28a745', '#6c757d']
            }]
        };

        new Chart(document.getElementById('supplierDebtsChart'), {
            type: 'bar',
            data: supplierDebtsData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Stok Giriş/Çıkış Karşılaştırma Grafiği
        const stockComparisonData = {
            labels: ['Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık'],
            datasets: [{
                label: 'Stok Girişi (Alış)',
                data: [
                    <?= json_encode($dashboardStats['stok_hareket']['aylik_giris'] ?? array_fill(0, 12, 0)) ?>
                ].flat(),
                backgroundColor: 'rgba(40, 167, 69, 0.8)',
                borderColor: 'rgba(40, 167, 69, 1)',
                borderWidth: 1
            }, {
                label: 'Satış (Çıkış)',
                data: [
                    <?= json_encode($dashboardStats['stok_hareket']['aylik_cikis'] ?? array_fill(0, 12, 0)) ?>
                ].flat(),
                backgroundColor: 'rgba(220, 53, 69, 0.8)',
                borderColor: 'rgba(220, 53, 69, 1)',
                borderWidth: 1
            }]
        };

        const stockComparisonChart = new Chart(document.getElementById('stockComparisonChart'), {
            type: 'bar',
            data: stockComparisonData,
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top'
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.dataset.label + ': ₺' + context.parsed.y.toLocaleString('tr-TR', {minimumFractionDigits: 2});
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '₺' + value.toLocaleString('tr-TR');
                            }
                        }
                    }
                }
            }
        });

        // Stok karşılaştırma güncelleme fonksiyonu
        function updateStockComparison(period) {
            // Aktif buton güncelleme
            document.querySelectorAll('.btn-group .btn').forEach(btn => btn.classList.remove('active'));
            event.target.classList.add('active');
            
            // AJAX ile veri güncelleme (gelecekte eklenebilir)
            console.log('Stok karşılaştırma güncellendi:', period);
        }
    </script>
</div>

<?php include '../includes/footer.php'; ?>