<?php

require_once 'error_handler.php';
require_once 'security.php';
require_once __DIR__ . '/../config/database.php';

class UserManager {
    private $db;
    
    public function __construct() {
        $database = Database::getInstance();
        $this->db = $database->getConnection();
    }
    
    public function createUser($kullanici_adi, $sifre, $ad_soyad, $rol_id, $aktif = 1) {
        try {
            $this->db->beginTransaction();
            
            $stmt = $this->db->prepare("SELECT kullanici_id FROM kullanicilar WHERE kullanici_adi = ?");
            $stmt->execute([$kullanici_adi]);
            
            if ($stmt->fetch()) {
                throw new Exception("Bu kullanıcı adı zaten kullanılıyor");
            }
            
            $sifre_hash = Security::hashPassword($sifre);
            
            $stmt = $this->db->prepare("
                INSERT INTO kullanicilar (kullanici_adi, sifre_hash, ad_soyad, rol_id, aktif) 
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([$kullanici_adi, $sifre_hash, $ad_soyad, $rol_id, $aktif]);
            
            $kullanici_id = $this->db->lastInsertId();
            
            $this->db->commit();
            
            ErrorHandler::logError("Yeni kullanıcı oluşturuldu: {$kullanici_adi}", 'user_create', [
                'kullanici_id' => $kullanici_id,
                'rol_id' => $rol_id,
                'olusturan' => $_SESSION['user_id'] ?? 'system'
            ]);
            
            return [
                'success' => true,
                'message' => 'Kullanıcı başarıyla oluşturuldu',
                'kullanici_id' => $kullanici_id
            ];
            
        } catch (Exception $e) {
            $this->db->rollBack();
            ErrorHandler::logError('user_create', $e->getMessage(), ['kullanici_adi' => $kullanici_adi]);
            return ['success' => false, 'message' => 'Kullanıcı oluşturulurken hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function updateUser($kullanici_id, $kullanici_adi, $ad_soyad, $rol_id, $aktif, $yeni_sifre = null) {
        try {
            $this->db->beginTransaction();
            
            $stmt = $this->db->prepare("SELECT kullanici_adi FROM kullanicilar WHERE kullanici_id = ?");
            $stmt->execute([$kullanici_id]);
            $mevcut_kullanici = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$mevcut_kullanici) {
                throw new Exception("Kullanıcı bulunamadı");
            }
            
            if ($mevcut_kullanici['kullanici_adi'] !== $kullanici_adi) {
                $stmt = $this->db->prepare("SELECT kullanici_id FROM kullanicilar WHERE kullanici_adi = ? AND kullanici_id != ?");
                $stmt->execute([$kullanici_adi, $kullanici_id]);
                
                if ($stmt->fetch()) {
                    throw new Exception("Bu kullanıcı adı zaten kullanılıyor");
                }
            }
            
            if ($yeni_sifre) {
                $sifre_hash = Security::hashPassword($yeni_sifre);
                $stmt = $this->db->prepare("
                    UPDATE kullanicilar 
                    SET kullanici_adi = ?, sifre_hash = ?, ad_soyad = ?, rol_id = ?, aktif = ? 
                    WHERE kullanici_id = ?
                ");
                $stmt->execute([$kullanici_adi, $sifre_hash, $ad_soyad, $rol_id, $aktif, $kullanici_id]);
            } else {
                $stmt = $this->db->prepare("
                    UPDATE kullanicilar 
                    SET kullanici_adi = ?, ad_soyad = ?, rol_id = ?, aktif = ? 
                    WHERE kullanici_id = ?
                ");
                $stmt->execute([$kullanici_adi, $ad_soyad, $rol_id, $aktif, $kullanici_id]);
            }
            
            $this->db->commit();
            
            ErrorHandler::logError("Kullanıcı güncellendi: {$kullanici_adi}", 'user_update', [
                'kullanici_id' => $kullanici_id,
                'rol_id' => $rol_id,
                'guncelleyen' => $_SESSION['user_id'] ?? 'system'
            ]);
            
            return [
                'success' => true,
                'message' => 'Kullanıcı başarıyla güncellendi'
            ];
            
        } catch (Exception $e) {
            $this->db->rollBack();
            ErrorHandler::logError('user_update', $e->getMessage(), ['kullanici_id' => $kullanici_id]);
            return ['success' => false, 'message' => 'Kullanıcı güncellenirken hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function deleteUser($kullanici_id) {
        try {
            if ($kullanici_id == $_SESSION['user_id']) {
                throw new Exception("Kendi hesabınızı silemezsiniz");
            }
            
            $stmt = $this->db->prepare("SELECT kullanici_adi FROM kullanicilar WHERE kullanici_id = ?");
            $stmt->execute([$kullanici_id]);
            $kullanici = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$kullanici) {
                throw new Exception("Kullanıcı bulunamadı");
            }
            
            $stmt = $this->db->prepare("UPDATE kullanicilar SET aktif = 0 WHERE kullanici_id = ?");
            $stmt->execute([$kullanici_id]);
            
            ErrorHandler::logError("Kullanıcı silindi: {$kullanici['kullanici_adi']}", 'user_delete', [
                'kullanici_id' => $kullanici_id,
                'silen' => $_SESSION['user_id'] ?? 'system'
            ]);
            
            return [
                'success' => true,
                'message' => 'Kullanıcı başarıyla silindi'
            ];
            
        } catch (Exception $e) {
            ErrorHandler::logError('user_delete', $e->getMessage(), ['kullanici_id' => $kullanici_id]);
            return ['success' => false, 'message' => 'Kullanıcı silinirken hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function getAllUsers($aktif_only = false) {
        try {
            $where_clause = $aktif_only ? "WHERE k.aktif = 1" : "";
            
            $stmt = $this->db->prepare("
                SELECT k.kullanici_id, k.kullanici_adi, k.ad_soyad, k.rol_id, k.aktif,
                       r.rol_adi, r.aciklama as rol_aciklama,
                       k.olusturulma_tarihi
                FROM kullanicilar k
                LEFT JOIN roller r ON k.rol_id = r.rol_id
                {$where_clause}
                ORDER BY k.kullanici_adi
            ");
            $stmt->execute();
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } catch (Exception $e) {
            ErrorHandler::logError('user_list', $e->getMessage());
            return [];
        }
    }
    
    public function getUserById($kullanici_id) {
        try {
            $stmt = $this->db->prepare("
                SELECT k.kullanici_id, k.kullanici_adi, k.ad_soyad, k.rol_id, k.aktif,
                       r.rol_adi, r.aciklama as rol_aciklama,
                       k.olusturulma_tarihi
                FROM kullanicilar k
                LEFT JOIN roller r ON k.rol_id = r.rol_id
                WHERE k.kullanici_id = ?
            ");
            $stmt->execute([$kullanici_id]);
            
            return $stmt->fetch(PDO::FETCH_ASSOC);
            
        } catch (Exception $e) {
            ErrorHandler::logError('user_get', $e->getMessage(), ['kullanici_id' => $kullanici_id]);
            return null;
        }
    }
    
    public function getAllRoles() {
        try {
            $stmt = $this->db->prepare("SELECT rol_id, rol_adi, aciklama FROM roller ORDER BY rol_id");
            $stmt->execute();
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } catch (Exception $e) {
            ErrorHandler::logError('roles_get', $e->getMessage());
            return [];
        }
    }
    
    public function changePassword($kullanici_id, $eski_sifre, $yeni_sifre) {
        try {
            $stmt = $this->db->prepare("SELECT sifre_hash FROM kullanicilar WHERE kullanici_id = ?");
            $stmt->execute([$kullanici_id]);
            $kullanici = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$kullanici) {
                throw new Exception("Kullanıcı bulunamadı");
            }
            
            if (!Security::verifyPassword($eski_sifre, $kullanici['sifre_hash'])) {
                throw new Exception("Mevcut şifre hatalı");
            }
            
            $yeni_sifre_hash = Security::hashPassword($yeni_sifre);
            $stmt = $this->db->prepare("UPDATE kullanicilar SET sifre_hash = ? WHERE kullanici_id = ?");
            $stmt->execute([$yeni_sifre_hash, $kullanici_id]);
            
            ErrorHandler::logError("Şifre değiştirildi", 'password_change', [
                'kullanici_id' => $kullanici_id
            ]);
            
            return [
                'success' => true,
                'message' => 'Şifre başarıyla değiştirildi'
            ];
            
        } catch (Exception $e) {
            ErrorHandler::logError('password_change', $e->getMessage(), ['kullanici_id' => $kullanici_id]);
            return ['success' => false, 'message' => 'Şifre değiştirilirken hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function getUserStats() {
        try {
            $stats = [];
            
            $stmt = $this->db->prepare("SELECT COUNT(*) as toplam FROM kullanicilar");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['toplam_kullanici'] = $result['toplam'];
            
            $stmt = $this->db->prepare("SELECT COUNT(*) as aktif FROM kullanicilar WHERE aktif = 1");
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['aktif_kullanici'] = $result['aktif'];
            
            $stmt = $this->db->prepare("
                SELECT r.rol_adi, COUNT(k.kullanici_id) as sayi 
                FROM roller r 
                LEFT JOIN kullanicilar k ON r.rol_id = k.rol_id AND k.aktif = 1
                GROUP BY r.rol_id, r.rol_adi
                ORDER BY r.rol_id
            ");
            $stmt->execute();
            $stats['rol_dagilimi'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return $stats;
            
        } catch (Exception $e) {
            ErrorHandler::logError('user_stats', $e->getMessage());
            return [];
        }
    }
}
?>