<?php

require_once __DIR__ . '/../config/database.php';

/**
 * TransactionManager Class
 * Gelişmiş transaction yönetimi sağlayan sınıf
 * Nested transaction ve savepoint desteği ile
 */
class TransactionManager {
    private $database;
    private $transactionStack = [];
    private $savepointCounter = 0;
    private $logFile;
    
    public function __construct($database = null) {
        $this->database = $database ?? Database::getInstance();
        $this->logFile = __DIR__ . '/../logs/transaction.log';
        
        $logDir = dirname($this->logFile);
        if (!file_exists($logDir)) {
            mkdir($logDir, 0755, true);
        }
    }
    
    /**
     * Transaction başlat (nested transaction desteği)
     * @return bool Başarılı mı
     */
    public function begin(): bool {
        try {
            // Eğer zaten transaction içindeysek, savepoint kullan
            if ($this->isActive()) {
                $savepointName = 'sp_' . (++$this->savepointCounter);
                $this->database->exec("SAVEPOINT $savepointName");
                $this->transactionStack[] = $savepointName;
                $this->log("Savepoint created: $savepointName");
                return true;
            }
            
            // İlk transaction'ı başlat
            $result = $this->database->beginTransaction();
            if ($result) {
                $this->transactionStack[] = 'main';
                $this->log("Transaction started");
            }
            return $result;
            
        } catch (Exception $e) {
            $this->log("Transaction begin error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Transaction commit et
     * @return bool Başarılı mı
     */
    public function commit(): bool {
        try {
            if (empty($this->transactionStack)) {
                $this->log("No active transaction to commit", 'WARNING');
                return false;
            }
            
            $current = array_pop($this->transactionStack);
            
            // Eğer savepoint ise, sadece stack'ten çıkar
            if ($current !== 'main') {
                $this->log("Savepoint released: $current");
                // MySQL'de savepoint release otomatik
                return true;
            }
            
            // Ana transaction'ı commit et
            $result = $this->database->commit();
            if ($result) {
                $this->log("Transaction committed");
                $this->savepointCounter = 0; // Reset counter
            }
            return $result;
            
        } catch (Exception $e) {
            $this->log("Transaction commit error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Transaction rollback yap
     * @return bool Başarılı mı
     */
    public function rollback(): bool {
        try {
            if (empty($this->transactionStack)) {
                $this->log("No active transaction to rollback", 'WARNING');
                return false;
            }
            
            $current = array_pop($this->transactionStack);
            
            // Eğer savepoint ise, sadece o savepoint'e geri dön
            if ($current !== 'main') {
                $this->database->exec("ROLLBACK TO SAVEPOINT $current");
                $this->log("Rolled back to savepoint: $current");
                return true;
            }
            
            // Ana transaction'ı rollback et
            $result = $this->database->rollback();
            if ($result) {
                $this->log("Transaction rolled back");
                $this->savepointCounter = 0; // Reset counter
            }
            return $result;
            
        } catch (Exception $e) {
            $this->log("Transaction rollback error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Savepoint oluştur
     * @param string $name Savepoint adı
     * @return bool Başarılı mı
     */
    public function createSavepoint(string $name): bool {
        try {
            if (!$this->isActive()) {
                $this->log("Cannot create savepoint: No active transaction", 'ERROR');
                return false;
            }
            
            $this->database->exec("SAVEPOINT $name");
            $this->transactionStack[] = $name;
            $this->log("Savepoint created: $name");
            return true;
            
        } catch (Exception $e) {
            $this->log("Savepoint creation error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Savepoint'e geri dön
     * @param string $name Savepoint adı
     * @return bool Başarılı mı
     */
    public function rollbackToSavepoint(string $name): bool {
        try {
            if (!$this->isActive()) {
                $this->log("Cannot rollback to savepoint: No active transaction", 'ERROR');
                return false;
            }
            
            $this->database->exec("ROLLBACK TO SAVEPOINT $name");
            
            // Stack'ten savepoint'i ve sonrasını temizle
            $index = array_search($name, $this->transactionStack);
            if ($index !== false) {
                $this->transactionStack = array_slice($this->transactionStack, 0, $index + 1);
            }
            
            $this->log("Rolled back to savepoint: $name");
            return true;
            
        } catch (Exception $e) {
            $this->log("Savepoint rollback error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Transaction içinde mi kontrol et
     * @return bool Transaction aktif mi
     */
    public function isActive(): bool {
        return !empty($this->transactionStack) && $this->database->inTransaction();
    }
    
    /**
     * Güvenli transaction wrapper
     * Callback fonksiyonunu transaction içinde çalıştırır
     * Hata durumunda otomatik rollback yapar
     * 
     * @param callable $callback Çalıştırılacak fonksiyon
     * @return mixed Callback'in dönüş değeri
     * @throws Exception Callback içinde oluşan hatalar
     */
    public function execute(callable $callback) {
        $wasActive = $this->isActive();
        
        try {
            // Transaction başlat (nested ise savepoint kullanır)
            if (!$wasActive) {
                $this->begin();
            }
            
            // Callback'i çalıştır
            $result = $callback($this->database);
            
            // Sadece biz başlattıysak commit et
            if (!$wasActive) {
                $this->commit();
            }
            
            return $result;
            
        } catch (Exception $e) {
            // Sadece biz başlattıysak rollback yap
            if (!$wasActive && $this->isActive()) {
                $this->rollback();
            }
            
            $this->log("Transaction execute error: " . $e->getMessage(), 'ERROR');
            throw $e;
        }
    }
    
    /**
     * Transaction durumunu al
     * @return array Transaction durumu
     */
    public function getStatus(): array {
        return [
            'active' => $this->isActive(),
            'stack_depth' => count($this->transactionStack),
            'stack' => $this->transactionStack,
            'savepoint_counter' => $this->savepointCounter
        ];
    }
    
    /**
     * Tüm transaction'ları zorla kapat (acil durum)
     * @return bool Başarılı mı
     */
    public function forceClose(): bool {
        try {
            if ($this->database->inTransaction()) {
                $this->database->rollback();
                $this->log("Force closed all transactions", 'WARNING');
            }
            
            $this->transactionStack = [];
            $this->savepointCounter = 0;
            return true;
            
        } catch (Exception $e) {
            $this->log("Force close error: " . $e->getMessage(), 'ERROR');
            return false;
        }
    }
    
    /**
     * Log mesajı yaz
     * @param string $message Mesaj
     * @param string $level Log seviyesi
     */
    private function log(string $message, string $level = 'INFO'): void {
        $timestamp = date('Y-m-d H:i:s');
        $stackInfo = json_encode($this->transactionStack);
        $logMessage = "[$timestamp] [$level] $message | Stack: $stackInfo\n";
        file_put_contents($this->logFile, $logMessage, FILE_APPEND);
    }
}

?>
