<?php

require_once 'error_handler.php';
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/transaction_manager.php';

class StockManager {
    private $database;
    private $transactionManager;
    
    public function __construct() {
        $this->database = Database::getInstance();
        $this->transactionManager = new TransactionManager($this->database);
    }
    
    public function stockEntry($urun_id, $miktar, $birim_fiyat = null, $aciklama = '', $referans_no = '') {
        try {
            return $this->transactionManager->execute(function($db) use ($urun_id, $miktar, $birim_fiyat, $aciklama, $referans_no) {
                $urun = $this->database->fetch("SELECT stok_miktari, urun_adi, alis_fiyati FROM urunler WHERE urun_id = ?", [$urun_id]);
                
                if (!$urun) {
                    throw new Exception("Ürün bulunamadı");
                }
                
                // Birim fiyat belirtilmemişse ürünün alış fiyatını kullan
                if ($birim_fiyat === null) {
                    $birim_fiyat = $urun['alis_fiyati'];
                }
                
                $onceki_stok = $urun['stok_miktari'];
                $yeni_stok = $onceki_stok + $miktar;
                $toplam_tutar = $birim_fiyat ? ($miktar * $birim_fiyat) : null;
                
                $this->database->execute("
                    INSERT INTO stok_hareketleri 
                    (urun_id, hareket_tipi, miktar, onceki_stok, yeni_stok, birim_fiyat, toplam_tutar, aciklama, referans_no, kullanici_id) 
                    VALUES (?, 'giris', ?, ?, ?, ?, ?, ?, ?, ?)
                ", [
                    $urun_id, $miktar, $onceki_stok, $yeni_stok, 
                    $birim_fiyat, $toplam_tutar, $aciklama, $referans_no, $_SESSION['user_id']
                ]);
                
                $this->database->execute("UPDATE urunler SET stok_miktari = ? WHERE urun_id = ?", [$yeni_stok, $urun_id]);
                
                ErrorHandler::logError("Stok girişi: {$urun['urun_adi']} - Miktar: {$miktar}", 'stock_entry', [
                    'urun_id' => $urun_id,
                    'miktar' => $miktar,
                    'onceki_stok' => $onceki_stok,
                    'yeni_stok' => $yeni_stok
                ]);
                
                return [
                    'success' => true,
                    'message' => 'Stok girişi başarıyla kaydedildi',
                    'onceki_stok' => $onceki_stok,
                    'yeni_stok' => $yeni_stok
                ];
            });
            
        } catch (Exception $e) {
            ErrorHandler::logError('stock_entry', $e->getMessage(), ['urun_id' => $urun_id, 'miktar' => $miktar]);
            return ['success' => false, 'message' => 'Stok girişi sırasında hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function stockExit($urun_id, $miktar, $birim_fiyat = null, $aciklama = '', $referans_no = '', $satis_id = null) {
        try {
            return $this->transactionManager->execute(function($db) use ($urun_id, $miktar, $birim_fiyat, $aciklama, $referans_no, $satis_id) {
                $urun = $this->database->fetch("SELECT stok_miktari, urun_adi, satis_fiyati FROM urunler WHERE urun_id = ?", [$urun_id]);
                
                if (!$urun) {
                    throw new Exception("Ürün bulunamadı");
                }
                
                // Birim fiyat belirtilmemişse ürünün satış fiyatını kullan
                if ($birim_fiyat === null) {
                    $birim_fiyat = $urun['satis_fiyati'];
                }
                
                $onceki_stok = $urun['stok_miktari'];
                
                if ($onceki_stok < $miktar) {
                    throw new Exception("Yetersiz stok! Mevcut stok: {$onceki_stok}");
                }
                
                $yeni_stok = $onceki_stok - $miktar;
                $toplam_tutar = $birim_fiyat ? ($miktar * $birim_fiyat) : null;
                
                $this->database->execute("
                    INSERT INTO stok_hareketleri 
                    (urun_id, hareket_tipi, miktar, onceki_stok, yeni_stok, birim_fiyat, toplam_tutar, aciklama, referans_no, satis_id, kullanici_id) 
                    VALUES (?, 'cikis', ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ", [
                    $urun_id, $miktar, $onceki_stok, $yeni_stok, 
                    $birim_fiyat, $toplam_tutar, $aciklama, $referans_no, $satis_id, $_SESSION['user_id']
                ]);
                
                $this->database->execute("UPDATE urunler SET stok_miktari = ? WHERE urun_id = ?", [$yeni_stok, $urun_id]);
                
                ErrorHandler::logError("Stok çıkışı: {$urun['urun_adi']} - Miktar: {$miktar}", 'stock_exit', [
                    'urun_id' => $urun_id,
                    'miktar' => $miktar,
                    'onceki_stok' => $onceki_stok,
                    'yeni_stok' => $yeni_stok
                ]);
                
                return [
                    'success' => true,
                    'message' => 'Stok çıkışı başarıyla kaydedildi',
                    'onceki_stok' => $onceki_stok,
                    'yeni_stok' => $yeni_stok
                ];
            });
            
        } catch (Exception $e) {
            ErrorHandler::logError('stock_exit', $e->getMessage(), ['urun_id' => $urun_id, 'miktar' => $miktar]);
            return ['success' => false, 'message' => 'Stok çıkışı sırasında hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function startStockCount($sayim_adi, $aciklama = '') {
        try {
            return $this->database->safeTransaction(function($db) use ($sayim_adi, $aciklama) {
                $this->database->execute("
                    INSERT INTO stok_sayim (sayim_tarihi, sayim_adi, aciklama, kullanici_id) 
                    VALUES (CURDATE(), ?, ?, ?)
                ", [$sayim_adi, $aciklama, $_SESSION['user_id']]);
                
                $sayim_id = $this->database->lastInsertId();
                
                $this->database->execute("
                    INSERT INTO stok_sayim_detay (sayim_id, urun_id, sistem_stok)
                    SELECT ?, urun_id, stok_miktari FROM urunler
                ", [$sayim_id]);
                
                ErrorHandler::logError("Stok sayımı başlatıldı: {$sayim_adi}", 'stock_count', ['sayim_id' => $sayim_id]);
                
                return [
                    'success' => true,
                    'message' => 'Stok sayımı başarıyla başlatıldı',
                    'sayim_id' => $sayim_id
                ];
            });
            
        } catch (Exception $e) {
            ErrorHandler::logError('stock_count', $e->getMessage(), ['sayim_adi' => $sayim_adi]);
            return ['success' => false, 'message' => 'Stok sayımı başlatılırken hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function updateStockCount($detay_id, $sayilan_stok, $aciklama = '') {
        try {
            $detay = $this->database->fetch("
                SELECT ssd.*, u.urun_adi 
                FROM stok_sayim_detay ssd 
                JOIN urunler u ON ssd.urun_id = u.urun_id 
                WHERE ssd.detay_id = ?
            ", [$detay_id]);
            
            if (!$detay) {
                throw new Exception("Sayım detayı bulunamadı");
            }
            
            $fark = $sayilan_stok - $detay['sistem_stok'];
            
            $this->database->execute("
                UPDATE stok_sayim_detay 
                SET sayilan_stok = ?, fark = ?, aciklama = ? 
                WHERE detay_id = ?
            ", [$sayilan_stok, $fark, $aciklama, $detay_id]);
            
            return [
                'success' => true,
                'message' => 'Sayım güncellendi',
                'fark' => $fark
            ];
            
        } catch (Exception $e) {
            ErrorHandler::logError('stock_count_update', $e->getMessage(), ['detay_id' => $detay_id]);
            return ['success' => false, 'message' => 'Sayım güncellenirken hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function completeStockCount($sayim_id) {
        try {
            $this->db->beginTransaction();
            
            $stmt = $this->db->prepare("
                SELECT ssd.*, u.urun_adi 
                FROM stok_sayim_detay ssd 
                JOIN urunler u ON ssd.urun_id = u.urun_id 
                WHERE ssd.sayim_id = ? AND ssd.sayilan_stok IS NOT NULL
            ");
            $stmt->execute([$sayim_id]);
            $detaylar = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($detaylar as $detay) {
                if ($detay['fark'] != 0) {
                    $hareket_tipi = $detay['fark'] > 0 ? 'giris' : 'cikis';
                    $miktar = abs($detay['fark']);
                    $yeni_stok = $detay['sayilan_stok'];
                    
                    $stmt = $this->db->prepare("
                        INSERT INTO stok_hareketleri 
                        (urun_id, hareket_tipi, miktar, onceki_stok, yeni_stok, aciklama, kullanici_id) 
                        VALUES (?, 'sayim', ?, ?, ?, ?, ?)
                    ");
                    $stmt->execute([
                        $detay['urun_id'], $detay['fark'], $detay['sistem_stok'], 
                        $yeni_stok, "Sayım düzeltmesi: {$detay['aciklama']}", $_SESSION['user_id']
                    ]);
                    
                    $stmt = $this->db->prepare("UPDATE urunler SET stok_miktari = ? WHERE urun_id = ?");
                    $stmt->execute([$yeni_stok, $detay['urun_id']]);
                }
            }
            
            $stmt = $this->db->prepare("
                UPDATE stok_sayim 
                SET durum = 'tamamlandi', tamamlanma_tarihi = NOW() 
                WHERE sayim_id = ?
            ");
            $stmt->execute([$sayim_id]);
            
            $this->db->commit();
            
            ErrorHandler::logInfo('stock_count_complete', "Stok sayımı tamamlandı", ['sayim_id' => $sayim_id]);
            
            return [
                'success' => true,
                'message' => 'Stok sayımı başarıyla tamamlandı',
                'updated_count' => count($detaylar)
            ];
            
        } catch (Exception $e) {
            $this->db->rollBack();
            ErrorHandler::logError('stock_count_complete', $e->getMessage(), ['sayim_id' => $sayim_id]);
            return ['success' => false, 'message' => 'Stok sayımı tamamlanırken hata oluştu: ' . $e->getMessage()];
        }
    }
    
    public function getStockStatus($filter = []) {
        try {
            $where_conditions = [];
            $params = [];
            
            // Stok durumu filtresi için özel kontrol
            if (!empty($filter['stok_durumu'])) {
                switch($filter['stok_durumu']) {
                    case 'Kritik':
                        $where_conditions[] = "u.stok_miktari <= 10";
                        break;
                    case 'Düşük':
                        $where_conditions[] = "u.stok_miktari > 10 AND u.stok_miktari <= 50";
                        break;
                    case 'Normal':
                        $where_conditions[] = "u.stok_miktari > 50 AND u.stok_miktari <= 100";
                        break;
                    case 'Yüksek':
                        $where_conditions[] = "u.stok_miktari > 100";
                        break;
                }
            }
            
            if (!empty($filter['urun_adi'])) {
                $where_conditions[] = "u.urun_adi LIKE ?";
                $params[] = '%' . $filter['urun_adi'] . '%';
            }
            
            $where_clause = !empty($where_conditions) ? 'WHERE ' . implode(' AND ', $where_conditions) : '';
            
            $sql = "
                SELECT 
                    u.urun_id,
                    u.urun_adi,
                    u.barkod,
                    u.stok_miktari,
                    COALESCE(sus.minimum_stok, 0) as minimum_stok,
                    u.alis_fiyati,
                    u.satis_fiyati,
                    (u.stok_miktari * u.alis_fiyati) as toplam_alis_degeri,
                    (u.stok_miktari * u.satis_fiyati) as toplam_satis_degeri,
                    CASE 
                        WHEN u.stok_miktari <= COALESCE(sus.kritik_seviye, 5) THEN 'Kritik'
                        WHEN u.stok_miktari <= COALESCE(sus.minimum_stok, 10) THEN 'Düşük'
                        WHEN u.stok_miktari <= 100 THEN 'Normal'
                        ELSE 'Yüksek'
                    END as stok_durumu
                FROM urunler u
                LEFT JOIN stok_uyari_seviyeleri sus ON u.urun_id = sus.urun_id
                {$where_clause}
                ORDER BY 
                    CASE 
                        WHEN u.stok_miktari <= 10 THEN 1
                        WHEN u.stok_miktari <= 50 THEN 2
                        WHEN u.stok_miktari <= 100 THEN 3
                        ELSE 4
                    END,
                    u.urun_adi
            ";
            
            return $this->database->fetchAll($sql, $params);
            
        } catch (Exception $e) {
            ErrorHandler::logError('stock_status', $e->getMessage(), $filter);
            return [];
        }
    }
    
    public function getStockHistory($urun_id = null, $limit = 100) {
        try {
            $where_clause = $urun_id ? 'WHERE sh.urun_id = ?' : '';
            $params = $urun_id ? [$urun_id] : [];
            
            $sql = "
                SELECT sh.*, u.urun_adi, k.kullanici_adi 
                FROM stok_hareketleri sh
                JOIN urunler u ON sh.urun_id = u.urun_id
                JOIN kullanicilar k ON sh.kullanici_id = k.kullanici_id
                {$where_clause}
                ORDER BY sh.tarih DESC
                LIMIT {$limit}
            ";
            
            return $this->database->fetchAll($sql, $params);
            
        } catch (Exception $e) {
            ErrorHandler::logError('stock_history', $e->getMessage(), ['urun_id' => $urun_id]);
            return [];
        }
    }
    
    public function getCriticalStockAlerts() {
        try {
            $sql = "
                SELECT 
                    u.urun_id,
                    u.urun_adi,
                    u.barkod,
                    u.stok_miktari,
                    u.minimum_stok,
                    u.alis_fiyati,
                    u.satis_fiyati,
                    (u.stok_miktari * u.alis_fiyati) as toplam_alis_degeri,
                    (u.stok_miktari * u.satis_fiyati) as toplam_satis_degeri,
                    CASE 
                        WHEN u.stok_miktari <= 10 THEN CAST('Kritik' AS CHAR CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci)
                        WHEN u.stok_miktari <= 50 THEN CAST('Düşük' AS CHAR CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci)
                        ELSE CAST('Normal' AS CHAR CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci)
                    END as stok_durumu
                FROM urunler u
                WHERE u.stok_miktari <= 50
                ORDER BY 
                    CASE 
                        WHEN u.stok_miktari <= 10 THEN 1
                        WHEN u.stok_miktari <= 50 THEN 2
                    END,
                    u.stok_miktari ASC
            ";
            
            return $this->database->fetchAll($sql);
            
        } catch (Exception $e) {
            ErrorHandler::logError('critical_stock', $e->getMessage());
            return [];
        }
    }
    
    public function updateStockLevels($urun_id, $minimum_stok, $maksimum_stok = null, $kritik_seviye = null) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO stok_uyari_seviyeleri (urun_id, minimum_stok, maksimum_stok, kritik_seviye)
                VALUES (?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE
                minimum_stok = VALUES(minimum_stok),
                maksimum_stok = VALUES(maksimum_stok),
                kritik_seviye = VALUES(kritik_seviye),
                guncelleme_tarihi = NOW()
            ");
            $stmt->execute([$urun_id, $minimum_stok, $maksimum_stok, $kritik_seviye]);
            
            return ['success' => true, 'message' => 'Stok seviyeleri güncellendi'];
            
        } catch (Exception $e) {
            ErrorHandler::logError('update_stock_levels', $e->getMessage(), [
                'urun_id' => $urun_id,
                'minimum_stok' => $minimum_stok
            ]);
            return ['success' => false, 'message' => 'Stok seviyeleri güncellenirken hata oluştu'];
        }
    }
}
?>