<?php

class SecurityScanner {
    private $vulnerabilities = [];
    private $scan_results = [];
    
    public function fullScan() {
        $this->vulnerabilities = [];
        $this->scan_results = [];
        
        $this->checkFilePermissions();
        $this->checkConfigurationSecurity();
        $this->checkDatabaseSecurity();
        $this->checkSessionSecurity();
        $this->checkInputValidation();
        $this->checkCSRFProtection();
        $this->checkXSSProtection();
        $this->checkSQLInjectionProtection();
        $this->checkFileUploadSecurity();
        $this->checkAuthenticationSecurity();
        $this->checkHTTPSConfiguration();
        $this->checkErrorHandling();
        
        return [
            'vulnerabilities' => $this->vulnerabilities,
            'scan_results' => $this->scan_results,
            'risk_level' => $this->calculateRiskLevel(),
            'scan_date' => date('Y-m-d H:i:s')
        ];
    }
    
    private function checkFilePermissions() {
        $critical_files = [
            'config/database.php',
            'config/config.php',
            'includes/security.php',
            '.htaccess'
        ];
        
        foreach ($critical_files as $file) {
            $full_path = __DIR__ . '/../' . $file;
            if (file_exists($full_path)) {
                $perms = fileperms($full_path);
                $octal_perms = substr(sprintf('%o', $perms), -4);
                
                if ($octal_perms > '0644') {
                    $this->addVulnerability('file_permissions', "Dosya izinleri çok geniş: $file ($octal_perms)", 'medium');
                }
            } else {
                $this->addVulnerability('missing_file', "Kritik dosya eksik: $file", 'high');
            }
        }
        
        $this->scan_results['file_permissions'] = 'completed';
    }
    
    private function checkConfigurationSecurity() {
        $php_settings = [
            'display_errors' => ['expected' => '0', 'risk' => 'medium'],
            'expose_php' => ['expected' => '0', 'risk' => 'low'],
            'allow_url_fopen' => ['expected' => '0', 'risk' => 'high'],
            'allow_url_include' => ['expected' => '0', 'risk' => 'high'],
            'register_globals' => ['expected' => '0', 'risk' => 'high']
        ];
        
        foreach ($php_settings as $setting => $config) {
            $value = ini_get($setting);
            if ($value != $config['expected']) {
                $this->addVulnerability('php_config', 
                    "Güvensiz PHP ayarı: $setting = $value (olması gereken: {$config['expected']})", 
                    $config['risk']);
            }
        }
        
        $test_dirs = ['uploads/', 'logs/', 'includes/'];
        foreach ($test_dirs as $dir) {
            $url = "http://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['REQUEST_URI']) . "/$dir";
            $headers = @get_headers($url);
            if ($headers && strpos($headers[0], '200') !== false) {
                $this->addVulnerability('directory_listing', 
                    "Dizin listeleme aktif: $dir", 'medium');
            }
        }
        
        $this->scan_results['configuration'] = 'completed';
    }
    
    private function checkDatabaseSecurity() {
        try {
            require_once __DIR__ . '/../config/database.php';
            $db = new Database();
            $pdo = $db->getConnection();
            
            $stmt = $pdo->query("SELECT user FROM mysql.user WHERE user IN ('root', 'admin', 'test')");
            $default_users = $stmt->fetchAll();
            
            if (count($default_users) > 0) {
                $this->addVulnerability('database_users', 
                    "Varsayılan veritabanı kullanıcıları mevcut", 'medium');
            }
            
            $stmt = $pdo->query("SELECT user FROM mysql.user WHERE authentication_string = '' OR password = ''");
            $empty_passwords = $stmt->fetchAll();
            
            if (count($empty_passwords) > 0) {
                $this->addVulnerability('database_passwords', 
                    "Boş şifreli veritabanı kullanıcıları mevcut", 'high');
            }
            
        } catch (Exception $e) {
            $this->addVulnerability('database_connection', 
                "Veritabanı güvenlik kontrolü yapılamadı: " . $e->getMessage(), 'low');
        }
        
        $this->scan_results['database'] = 'completed';
    }
    
    private function checkSessionSecurity() {
        $session_settings = [
            'session.cookie_httponly' => ['expected' => '1', 'risk' => 'medium'],
            'session.cookie_secure' => ['expected' => '1', 'risk' => 'medium'],
            'session.use_strict_mode' => ['expected' => '1', 'risk' => 'medium'],
            'session.cookie_samesite' => ['expected' => 'Strict', 'risk' => 'low']
        ];
        
        foreach ($session_settings as $setting => $config) {
            $value = ini_get($setting);
            if ($value != $config['expected']) {
                $this->addVulnerability('session_config', 
                    "Güvensiz session ayarı: $setting = $value", $config['risk']);
            }
        }
        
        $this->scan_results['session'] = 'completed';
    }
    
    private function checkInputValidation() {
        $test_inputs = [
            '<script>alert("xss")</script>',
            "'; DROP TABLE users; --",
            '../../../etc/passwd',
            '<?php phpinfo(); ?>',
            'javascript:alert("xss")'
        ];
        
        $this->scan_results['input_validation'] = 'manual_check_required';
    }
    
    private function checkCSRFProtection() {
        $forms_to_check = [
            'pages/login.php',
            'pages/products.php',
            'pages/customers.php',
            'pages/sales.php'
        ];
        
        $csrf_protected = 0;
        $total_forms = 0;
        
        foreach ($forms_to_check as $file) {
            $full_path = __DIR__ . '/../' . $file;
            if (file_exists($full_path)) {
                $content = file_get_contents($full_path);
                $total_forms++;
                
                if (strpos($content, 'csrf_token') !== false) {
                    $csrf_protected++;
                }
            }
        }
        
        if ($total_forms > 0 && $csrf_protected < $total_forms) {
            $this->addVulnerability('csrf_protection', 
                "CSRF koruması eksik: $csrf_protected/$total_forms form korunuyor", 'high');
        }
        
        $this->scan_results['csrf'] = 'completed';
    }
    
    private function checkXSSProtection() {
        $csp_header = false;
        if (function_exists('apache_response_headers')) {
            $headers = apache_response_headers();
            $csp_header = isset($headers['Content-Security-Policy']);
        }
        
        if (!$csp_header) {
            $this->addVulnerability('xss_protection', 
                "Content Security Policy (CSP) header eksik", 'medium');
        }
        
        if (!isset($_SERVER['HTTP_X_XSS_PROTECTION'])) {
            $this->addVulnerability('xss_protection', 
                "X-XSS-Protection header eksik", 'low');
        }
        
        $this->scan_results['xss'] = 'completed';
    }
    
    private function checkSQLInjectionProtection() {
        $php_files = glob(__DIR__ . '/../pages/*.php');
        $vulnerable_patterns = [
            '/mysql_query\s*\(\s*["\'].*\$/',
            '/mysqli_query\s*\(\s*.*\$/',
            '/query\s*\(\s*["\'].*\$/',
            '/SELECT.*\$.*FROM/',
            '/INSERT.*\$.*INTO/',
            '/UPDATE.*\$.*SET/',
            '/DELETE.*\$.*FROM/'
        ];
        
        $vulnerable_files = [];
        
        foreach ($php_files as $file) {
            $content = file_get_contents($file);
            foreach ($vulnerable_patterns as $pattern) {
                if (preg_match($pattern, $content)) {
                    $vulnerable_files[] = basename($file);
                    break;
                }
            }
        }
        
        if (!empty($vulnerable_files)) {
            $this->addVulnerability('sql_injection', 
                "Potansiyel SQL injection açığı: " . implode(', ', $vulnerable_files), 'high');
        }
        
        $this->scan_results['sql_injection'] = 'completed';
    }
    
    private function checkFileUploadSecurity() {
        $upload_dir = __DIR__ . '/../uploads/';
        
        if (!file_exists($upload_dir . '.htaccess')) {
            $this->addVulnerability('file_upload', 
                "Upload dizininde .htaccess dosyası eksik", 'high');
        }
        
        $php_files = glob($upload_dir . '**/*.php');
        if (!empty($php_files)) {
            $this->addVulnerability('file_upload', 
                "Upload dizininde PHP dosyaları mevcut", 'critical');
        }
        
        $this->scan_results['file_upload'] = 'completed';
    }
    
    private function checkAuthenticationSecurity() {
        $this->scan_results['authentication'] = 'manual_check_required';
    }
    
    private function checkHTTPSConfiguration() {
        if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
            $this->addVulnerability('https', 
                "HTTPS kullanılmıyor", 'medium');
        }
        
        $this->scan_results['https'] = 'completed';
    }
    
    private function checkErrorHandling() {
        if (ini_get('display_errors') == '1') {
            $this->addVulnerability('error_handling', 
                "Hata mesajları görünür durumda", 'medium');
        }
        
        $this->scan_results['error_handling'] = 'completed';
    }
    
    private function addVulnerability($type, $description, $risk_level) {
        $this->vulnerabilities[] = [
            'type' => $type,
            'description' => $description,
            'risk_level' => $risk_level,
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
    
    private function calculateRiskLevel() {
        $risk_scores = ['low' => 1, 'medium' => 3, 'high' => 7, 'critical' => 10];
        $total_score = 0;
        
        foreach ($this->vulnerabilities as $vuln) {
            $total_score += $risk_scores[$vuln['risk_level']] ?? 0;
        }
        
        if ($total_score >= 20) return 'critical';
        if ($total_score >= 10) return 'high';
        if ($total_score >= 5) return 'medium';
        return 'low';
    }
    
    public function generateHTMLReport($scan_results) {
        $html = '<div class="security-scan-report">';
        $html .= '<h3>Güvenlik Tarama Raporu</h3>';
        $html .= '<p><strong>Tarama Tarihi:</strong> ' . $scan_results['scan_date'] . '</p>';
        $html .= '<p><strong>Risk Seviyesi:</strong> <span class="badge bg-' . $this->getRiskColor($scan_results['risk_level']) . '">' . strtoupper($scan_results['risk_level']) . '</span></p>';
        
        if (empty($scan_results['vulnerabilities'])) {
            $html .= '<div class="alert alert-success">✅ Güvenlik açığı bulunamadı!</div>';
        } else {
            $html .= '<h4>Bulunan Güvenlik Açıkları (' . count($scan_results['vulnerabilities']) . ')</h4>';
            $html .= '<div class="table-responsive">';
            $html .= '<table class="table table-striped">';
            $html .= '<thead><tr><th>Tür</th><th>Açıklama</th><th>Risk</th><th>Zaman</th></tr></thead>';
            $html .= '<tbody>';
            
            foreach ($scan_results['vulnerabilities'] as $vuln) {
                $html .= '<tr>';
                $html .= '<td>' . htmlspecialchars($vuln['type']) . '</td>';
                $html .= '<td>' . htmlspecialchars($vuln['description']) . '</td>';
                $html .= '<td><span class="badge bg-' . $this->getRiskColor($vuln['risk_level']) . '">' . strtoupper($vuln['risk_level']) . '</span></td>';
                $html .= '<td>' . $vuln['timestamp'] . '</td>';
                $html .= '</tr>';
            }
            
            $html .= '</tbody></table>';
            $html .= '</div>';
        }
        
        $html .= '</div>';
        return $html;
    }
    
    private function getRiskColor($risk_level) {
        switch ($risk_level) {
            case 'critical': return 'danger';
            case 'high': return 'warning';
            case 'medium': return 'info';
            case 'low': return 'secondary';
            default: return 'light';
        }
    }
}
?>

