<?php

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/security.php';
require_once __DIR__ . '/error_handler.php';

class ReportManager {
    private $db;
    private $security;
    private $errorHandler;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->security = new Security();
        $this->errorHandler = new ErrorHandler();
    }
    
    public function getReportTemplates($reportType = null) {
        try {
            $sql = "SELECT * FROM rapor_sablonlari WHERE aktif = 1";
            $params = [];
            
            if ($reportType) {
                $sql .= " AND rapor_tipi = ?";
                $params[] = $reportType;
            }
            
            $sql .= " ORDER BY rapor_adi";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::getReportTemplates', $e->getMessage());
            return [];
        }
    }
    
    public function generateReport($reportId, $parameters = [], $userId = null) {
        try {
            $template = $this->getReportTemplate($reportId);
            if (!$template) {
                throw new Exception('Rapor şablonu bulunamadı');
            }
            
            $query = $template['rapor_sorgusu'];
            $reportParams = json_decode($template['rapor_parametreleri'], true) ?? [];
            
            $queryParams = [];
            foreach ($reportParams as $key => $defaultValue) {
                $queryParams[] = $parameters[$key] ?? $defaultValue;
            }
            
            $stmt = $this->db->prepare($query);
            $stmt->execute($queryParams);
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $this->saveReportHistory($reportId, $userId, $parameters, 'tamamlandi');
            
            return [
                'success' => true,
                'data' => $data,
                'template' => $template,
                'parameters' => $parameters
            ];
            
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::generateReport', $e->getMessage());
            $this->saveReportHistory($reportId, $userId, $parameters, 'hata', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    private function getReportTemplate($reportId) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM rapor_sablonlari WHERE rapor_id = ? AND aktif = 1");
            $stmt->execute([$reportId]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::getReportTemplate', $e->getMessage());
            return null;
        }
    }
    
    private function saveReportHistory($reportId, $userId, $parameters, $status, $errorMessage = null) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO rapor_gecmisi (rapor_id, kullanici_id, rapor_parametreleri, rapor_durumu, hata_mesaji)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $reportId,
                $userId,
                json_encode($parameters),
                $status,
                $errorMessage
            ]);
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::saveReportHistory', $e->getMessage());
        }
    }
    
    public function getWidgetData($widgetId) {
        try {
            $widget = $this->getWidget($widgetId);
            if (!$widget) {
                throw new Exception('Widget bulunamadı');
            }
            
            $stmt = $this->db->prepare($widget['widget_sorgusu']);
            $stmt->execute();
            $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return [
                'success' => true,
                'data' => $data,
                'widget' => $widget
            ];
            
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::getWidgetData', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    private function getWidget($widgetId) {
        try {
            $stmt = $this->db->prepare("SELECT * FROM rapor_widgets WHERE widget_id = ? AND aktif = 1");
            $stmt->execute([$widgetId]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::getWidget', $e->getMessage());
            return null;
        }
    }
    
    public function getAllWidgets() {
        try {
            $stmt = $this->db->prepare("SELECT * FROM rapor_widgets WHERE aktif = 1 ORDER BY widget_adi");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::getAllWidgets', $e->getMessage());
            return [];
        }
    }
    
    public function exportToExcel($reportData, $filename = null) {
        try {
            if (!$filename) {
                $filename = 'rapor_' . date('Y-m-d_H-i-s') . '.csv';
            }
            
            $output = fopen('php://temp', 'w');
            
            if (!empty($reportData['data'])) {
                $headers = array_keys($reportData['data'][0]);
                fputcsv($output, $headers);
                
                foreach ($reportData['data'] as $row) {
                    fputcsv($output, $row);
                }
            }
            
            rewind($output);
            $csvContent = stream_get_contents($output);
            fclose($output);
            
            return [
                'success' => true,
                'content' => $csvContent,
                'filename' => $filename,
                'mime_type' => 'text/csv'
            ];
            
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::exportToExcel', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    public function exportToPDF($reportData, $filename = null) {
        try {
            if (!$filename) {
                $filename = 'rapor_' . date('Y-m-d_H-i-s') . '.html';
            }
            
            $html = $this->generateHTMLReport($reportData);
            
            return [
                'success' => true,
                'content' => $html,
                'filename' => $filename,
                'mime_type' => 'text/html'
            ];
            
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::exportToPDF', $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
    
    private function generateHTMLReport($reportData) {
        $html = '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <title>' . htmlspecialchars($reportData['template']['rapor_adi']) . '</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 20px; }
                table { border-collapse: collapse; width: 100%; margin-top: 20px; }
                th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                th { background-color: #f2f2f2; font-weight: bold; }
                .header { margin-bottom: 20px; }
                .title { font-size: 24px; font-weight: bold; margin-bottom: 10px; }
                .date { color: #666; }
            </style>
        </head>
        <body>
            <div class="header">
                <div class="title">' . htmlspecialchars($reportData['template']['rapor_adi']) . '</div>
                <div class="date">Oluşturulma Tarihi: ' . date('d.m.Y H:i:s') . '</div>
            </div>';
        
        if (!empty($reportData['data'])) {
            $html .= '<table>';
            
            $html .= '<thead><tr>';
            foreach (array_keys($reportData['data'][0]) as $header) {
                $html .= '<th>' . htmlspecialchars($header) . '</th>';
            }
            $html .= '</tr></thead>';
            
            $html .= '<tbody>';
            foreach ($reportData['data'] as $row) {
                $html .= '<tr>';
                foreach ($row as $cell) {
                    $html .= '<td>' . htmlspecialchars($cell) . '</td>';
                }
                $html .= '</tr>';
            }
            $html .= '</tbody>';
            
            $html .= '</table>';
        } else {
            $html .= '<p>Rapor için veri bulunamadı.</p>';
        }
        
        $html .= '</body></html>';
        
        return $html;
    }
    
    public function getReportHistory($userId = null, $limit = 50) {
        try {
            $sql = "SELECT rg.*, rs.rapor_adi, k.kullanici_adi 
                    FROM rapor_gecmisi rg
                    LEFT JOIN rapor_sablonlari rs ON rg.rapor_id = rs.rapor_id
                    LEFT JOIN kullanicilar k ON rg.kullanici_id = k.kullanici_id";
            
            $params = [];
            if ($userId) {
                $sql .= " WHERE rg.kullanici_id = ?";
                $params[] = $userId;
            }
            
            $sql .= " ORDER BY rg.olusturulma_tarihi DESC LIMIT ?";
            $params[] = $limit;
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::getReportHistory', $e->getMessage());
            return [];
        }
    }
    
    public function getDashboardStats() {
        try {
            $stats = [];
            
            // Stok istatistikleri
            $stats['stok'] = $this->db->fetch("SELECT * FROM stok_durum_ozeti");
            
            // Cari hesap istatistikleri
            $stats['cari'] = $this->db->fetch("
                SELECT 
                    COUNT(*) as toplam_cari,
                    SUM(CASE WHEN cari_tipi = 'musteri' AND cari_bakiye > 0 THEN cari_bakiye ELSE 0 END) as musteri_alacak,
                    SUM(CASE WHEN cari_tipi = 'tedarikci' AND cari_bakiye > 0 THEN cari_bakiye ELSE 0 END) as tedarikci_borc,
                    COUNT(CASE WHEN cari_tipi = 'musteri' THEN 1 END) as toplam_musteri,
                    COUNT(CASE WHEN cari_tipi = 'tedarikci' THEN 1 END) as toplam_tedarikci,
                    COUNT(CASE WHEN cari_tipi = 'musteri' AND cari_bakiye > 0 THEN 1 END) as borclu_musteri,
                    COUNT(CASE WHEN cari_tipi = 'musteri' AND cari_bakiye < 0 THEN 1 END) as alacakli_musteri,
                    COUNT(CASE WHEN cari_tipi = 'tedarikci' AND cari_bakiye > 0 THEN 1 END) as borclu_tedarikci,
                    COUNT(CASE WHEN cari_tipi = 'tedarikci' AND cari_bakiye < 0 THEN 1 END) as alacakli_tedarikci
                FROM cariler
            ");
            
            // Rapor istatistikleri
            $stats['raporlar'] = $this->db->fetch("SELECT COUNT(*) as toplam FROM rapor_gecmisi WHERE olusturulma_tarihi >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
            
            // Stok hareket istatistikleri (giriş/çıkış karşılaştırması)
            $stockStats = $this->getStockMovementStats();
            $stats['stok_hareket'] = [
                'toplam_giris' => $stockStats['toplam_giris'],
                'toplam_cikis' => $stockStats['toplam_cikis'],
                'aylik_giris' => $stockStats['aylik_giris'],
                'aylik_cikis' => $stockStats['aylik_cikis']
            ];
            
            return $stats;
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::getDashboardStats', $e->getMessage());
            return [
                'stok' => [],
                'cari' => [],
                'raporlar' => [],
                'stok_hareket' => [
                    'toplam_giris' => 0,
                    'toplam_cikis' => 0,
                    'aylik_giris' => array_fill(0, 12, 0),
                    'aylik_cikis' => array_fill(0, 12, 0)
                ]
            ];
        }
    }
    
    private function getStockMovementStats() {
        try {
            $stats = [];
            
            // Toplam stok girişi (alış değeri)
            $result = $this->db->fetch("
                SELECT 
                    COALESCE(SUM(CASE WHEN hareket_tipi = 'giris' THEN miktar * birim_fiyat ELSE 0 END), 0) as toplam_giris,
                    COALESCE(SUM(CASE WHEN hareket_tipi = 'cikis' THEN miktar * birim_fiyat ELSE 0 END), 0) as toplam_cikis
                FROM stok_hareketleri 
                WHERE YEAR(tarih) = YEAR(CURDATE())
            ");
            $stats['toplam_giris'] = (float)($result['toplam_giris'] ?? 0);
            $stats['toplam_cikis'] = (float)($result['toplam_cikis'] ?? 0);
            
            // Aylık stok girişi
            $aylik_giris_data = $this->db->fetchAll("
                SELECT 
                    MONTH(tarih) as ay,
                    COALESCE(SUM(miktar * birim_fiyat), 0) as giris_degeri
                FROM stok_hareketleri 
                WHERE YEAR(tarih) = YEAR(CURDATE()) AND hareket_tipi = 'giris'
                GROUP BY MONTH(tarih)
                ORDER BY ay
            ");
            $aylik_giris = array_fill(0, 12, 0);
            foreach ($aylik_giris_data as $row) {
                $aylik_giris[$row['ay'] - 1] = (float)$row['giris_degeri'];
            }
            $stats['aylik_giris'] = $aylik_giris;
            
            // Aylık satış (çıkış)
            $aylik_cikis_data = $this->db->fetchAll("
                SELECT 
                    MONTH(s.satis_tarihi) as ay,
                    COALESCE(SUM(s.toplam_tutar), 0) as satis_degeri
                FROM satislar s
                WHERE YEAR(s.satis_tarihi) = YEAR(CURDATE())
                GROUP BY MONTH(s.satis_tarihi)
                ORDER BY ay
            ");
            $aylik_cikis = array_fill(0, 12, 0);
            foreach ($aylik_cikis_data as $row) {
                $aylik_cikis[$row['ay'] - 1] = (float)$row['satis_degeri'];
            }
            $stats['aylik_cikis'] = $aylik_cikis;
            
            return $stats;
        } catch (Exception $e) {
            $this->errorHandler->logError('ReportManager::getStockMovementStats', $e->getMessage());
            return [
                'toplam_giris' => 0,
                'toplam_cikis' => 0,
                'aylik_giris' => array_fill(0, 12, 0),
                'aylik_cikis' => array_fill(0, 12, 0)
            ];
        }
    }
}
?>