<?php

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/transaction_manager.php';

class BalanceManager {
    private $database;
    private $transactionManager;
    
    public function __construct($database = null) {
        if ($database !== null) {
            $this->database = $database;
        } else {
            $this->database = Database::getInstance();
        }
        $this->transactionManager = new TransactionManager($this->database);
    }
    
    public function calculateBalance($cari_id) {
        try {
            $sql = "SELECT COALESCE(SUM(
                        CASE 
                            WHEN hareket_tipi = 'borc' THEN tutar 
                            WHEN hareket_tipi = 'alacak' THEN -tutar 
                            ELSE 0 
                        END
                    ), 0) as bakiye
                    FROM cari_hareketler 
                    WHERE cari_id = ?";
            
            $result = $this->database->fetch($sql, [$cari_id]);
            return (float) $result['bakiye'];
            
        } catch (Exception $e) {
            error_log("Bakiye hesaplama hatası (Cari ID: $cari_id): " . $e->getMessage());
            return 0.0;
        }
    }
    
    public function updateBalance($cari_id) {
        try {
            return $this->database->safeTransaction(function($db) use ($cari_id) {
                $calculated_balance = $this->calculateBalance($cari_id);
                
                // Hem cari_bakiye hem de bakiye alanlarını güncelle (aynı değer)
                $sql = "UPDATE cariler SET cari_bakiye = ?, bakiye = ? WHERE cari_id = ?";
                $this->database->execute($sql, [$calculated_balance, $calculated_balance, $cari_id]);
                
                return true;
            });
            
        } catch (Exception $e) {
            error_log("Bakiye güncelleme hatası (Cari ID: $cari_id): " . $e->getMessage());
            return false;
        }
    }
    
    public function updateAllBalances() {
        try {
            $sql = "SELECT cari_id FROM cariler";
            $cariler = $this->database->fetchAll($sql);
            
            $updated_count = 0;
            $error_count = 0;
            
            foreach ($cariler as $cari) {
                if ($this->updateBalance($cari['cari_id'])) {
                    $updated_count++;
                } else {
                    $error_count++;
                }
            }
            
            return [
                'updated' => $updated_count,
                'errors' => $error_count,
                'total' => count($cariler)
            ];
            
        } catch (Exception $e) {
            error_log("Toplu bakiye güncelleme hatası: " . $e->getMessage());
            return ['updated' => 0, 'errors' => 1, 'total' => 0];
        }
    }
    
    public function checkBalanceConsistency() {
        try {
            $sql = "SELECT 
                        c.cari_id,
                        c.firma_adi,
                        c.cari_bakiye as stored_balance,
                        COALESCE(SUM(
                            CASE 
                                WHEN ch.hareket_tipi = 'borc' THEN ch.tutar 
                                WHEN ch.hareket_tipi = 'alacak' THEN -ch.tutar 
                                ELSE 0 
                            END
                        ), 0) as calculated_balance
                    FROM cariler c
                    LEFT JOIN cari_hareketler ch ON c.cari_id = ch.cari_id
                    GROUP BY c.cari_id, c.firma_adi, c.cari_bakiye
                    HAVING ABS(c.cari_bakiye - calculated_balance) > 0.01";
            
            return $this->database->fetchAll($sql);
            
        } catch (Exception $e) {
            error_log("Bakiye tutarlılık kontrolü hatası: " . $e->getMessage());
            return [];
        }
    }
    
    public function addMovementAndUpdateBalance($cari_id, $hareket_tipi, $tutar, $aciklama = null, $satis_id = null) {
        try {
            return $this->transactionManager->execute(function($db) use ($cari_id, $hareket_tipi, $tutar, $aciklama, $satis_id) {
                $sql = "INSERT INTO cari_hareketler (cari_id, hareket_tipi, tutar, aciklama, satis_id) 
                        VALUES (?, ?, ?, ?, ?)";
                $this->database->execute($sql, [$cari_id, $hareket_tipi, $tutar, $aciklama, $satis_id]);
                
                // Bakiye güncelle (aynı değer)
                $calculated_balance = $this->calculateBalance($cari_id);
                $sql = "UPDATE cariler SET cari_bakiye = ?, bakiye = ? WHERE cari_id = ?";
                $this->database->execute($sql, [$calculated_balance, $calculated_balance, $cari_id]);
                
                return true;
            });
            
        } catch (Exception $e) {
            error_log("Cari hareket ekleme hatası: " . $e->getMessage());
            throw $e;
        }
    }
    
    public function getBalanceSummary() {
        try {
            $sql = "SELECT 
                        COUNT(*) as toplam_cari,
                        SUM(CASE WHEN cari_bakiye > 0 THEN 1 ELSE 0 END) as borclu_cari,
                        SUM(CASE WHEN cari_bakiye < 0 THEN 1 ELSE 0 END) as alacakli_cari,
                        SUM(CASE WHEN cari_bakiye = 0 THEN 1 ELSE 0 END) as sifir_bakiye,
                        SUM(CASE WHEN cari_bakiye > 0 THEN cari_bakiye ELSE 0 END) as toplam_borc,
                        SUM(CASE WHEN cari_bakiye < 0 THEN ABS(cari_bakiye) ELSE 0 END) as toplam_alacak,
                        SUM(cari_bakiye) as net_bakiye
                    FROM cariler";
            
            return $this->database->fetch($sql);
            
        } catch (Exception $e) {
            error_log("Bakiye özeti hatası: " . $e->getMessage());
            return [];
        }
    }
    
    public function getTopDebtors($limit = 10) {
        try {
            $sql = "SELECT cari_id, firma_adi, cari_bakiye 
                    FROM cariler 
                    WHERE cari_bakiye > 0 
                    ORDER BY cari_bakiye DESC 
                    LIMIT ?";
            
            return $this->database->fetchAll($sql, [$limit]);
            
        } catch (Exception $e) {
            error_log("En yüksek borçlular listesi hatası: " . $e->getMessage());
            return [];
        }
    }
    
    public function getTopCreditors($limit = 10) {
        try {
            $sql = "SELECT cari_id, firma_adi, ABS(cari_bakiye) as alacak_tutari 
                    FROM cariler 
                    WHERE cari_bakiye < 0 
                    ORDER BY cari_bakiye ASC 
                    LIMIT ?";
            
            return $this->database->fetchAll($sql, [$limit]);
            
        } catch (Exception $e) {
            error_log("En yüksek alacaklılar listesi hatası: " . $e->getMessage());
            return [];
        }
    }
    
    public function runOptimization() {
        try {
            $report = [];
            
            $inconsistencies = $this->checkBalanceConsistency();
            $report['inconsistencies_found'] = count($inconsistencies);
            $report['inconsistent_accounts'] = $inconsistencies;
            
            $update_result = $this->updateAllBalances();
            $report['balance_update'] = $update_result;
            
            $final_inconsistencies = $this->checkBalanceConsistency();
            $report['final_inconsistencies'] = count($final_inconsistencies);
            
            $report['summary'] = $this->getBalanceSummary();
            
            return $report;
            
        } catch (Exception $e) {
            error_log("Veritabanı optimizasyonu hatası: " . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }
    
    public function checkSupplierDebtMechanism() {
        try {
            // Tedarikçi borçlanma mekanizması kontrolü
            $sql = "SELECT 
                        COUNT(*) as toplam_tedarikci,
                        SUM(CASE WHEN cari_bakiye > 0 THEN 1 ELSE 0 END) as borclu_tedarikci,
                        SUM(CASE WHEN cari_bakiye > 0 THEN cari_bakiye ELSE 0 END) as toplam_tedarikci_borcu
                    FROM cariler 
                    WHERE cari_tipi = 'tedarikci'";
            
            $summary = $this->database->fetch($sql);
            
            // Problemli tedarikçileri bul (negatif bakiyeli tedarikçiler)
            $problematic_sql = "SELECT 
                                    cari_id, 
                                    firma_adi, 
                                    cari_bakiye as bakiye,
                                    CASE 
                                        WHEN cari_bakiye < 0 THEN 'Negatif bakiye - Tedarikçi bizden alacaklı görünüyor'
                                        ELSE 'Normal'
                                    END as problem
                                FROM cariler 
                                WHERE cari_tipi = 'tedarikci' AND cari_bakiye < 0
                                ORDER BY cari_bakiye ASC";
            
            $problematic_suppliers = $this->database->fetchAll($problematic_sql);
            
            return [
                'toplam_tedarikci' => $summary['toplam_tedarikci'],
                'borclu_tedarikci' => $summary['borclu_tedarikci'],
                'toplam_tedarikci_borcu' => $summary['toplam_tedarikci_borcu'],
                'problematic_suppliers' => $problematic_suppliers
            ];
            
        } catch (Exception $e) {
            error_log("Tedarikçi borçlanma mekanizması kontrolü hatası: " . $e->getMessage());
            return [
                'toplam_tedarikci' => 0,
                'borclu_tedarikci' => 0,
                'toplam_tedarikci_borcu' => 0,
                'problematic_suppliers' => []
            ];
        }
    }
    
    /**
     * Merkezi Finansal İşlem Yönetimi
     * Tüm finansal işlemleri tek noktadan yönetir ve tutarlılığı sağlar
     */
    
    /**
     * Satış işlemi gerçekleştir
     * @param int $cari_id Müşteri ID
     * @param array $urunler Ürün listesi [['urun_id' => 1, 'miktar' => 2, 'fiyat' => 15.00], ...]
     * @param float $odenen_tutar Ödenen tutar
     * @param string $aciklama Açıklama
     * @return array İşlem sonucu
     */
    public function processSale($cari_id, $urunler, $odenen_tutar = 0, $aciklama = '') {
        try {
            $this->database->beginTransaction();
            
            // Toplam tutarı hesapla
            $toplam_tutar = 0;
            foreach ($urunler as $urun) {
                $toplam_tutar += $urun['miktar'] * $urun['fiyat'];
            }
            
            // Satış kaydını oluştur
            $sql = "INSERT INTO satislar (cari_id, toplam_tutar, odenen_tutar, kalan_tutar, satis_tarihi) 
                    VALUES (?, ?, ?, ?, NOW())";
            $this->database->execute($sql, [$cari_id, $toplam_tutar, $odenen_tutar, $toplam_tutar - $odenen_tutar]);
            $satis_id = $this->database->lastInsertId();
            
            // Satış detaylarını ekle
            foreach ($urunler as $urun) {
                $sql = "INSERT INTO satis_detay (satis_id, urun_id, miktar, birim_fiyat, toplam_fiyat) 
                        VALUES (?, ?, ?, ?, ?)";
                $this->database->execute($sql, [
                    $satis_id, 
                    $urun['urun_id'], 
                    $urun['miktar'], 
                    $urun['fiyat'], 
                    $urun['miktar'] * $urun['fiyat']
                ]);
                
                // Stok güncelle
                $sql = "UPDATE urunler SET stok_miktari = stok_miktari - ? WHERE urun_id = ?";
                $this->database->execute($sql, [$urun['miktar'], $urun['urun_id']]);
            }
            
            // Cari hareketleri ekle
            // Borç kaydı (toplam tutar)
            $this->addMovementAndUpdateBalance($cari_id, 'borc', $toplam_tutar, 
                "Satış No: $satis_id" . ($aciklama ? " - $aciklama" : ""), $satis_id);
            
            // Alacak kaydı (ödenen tutar, eğer varsa)
            if ($odenen_tutar > 0) {
                $this->addMovementAndUpdateBalance($cari_id, 'alacak', $odenen_tutar, 
                    "Satış No: $satis_id - Ödeme" . ($aciklama ? " - $aciklama" : ""), $satis_id);
            }
            
            $this->database->commit();
            
            return [
                'success' => true,
                'satis_id' => $satis_id,
                'toplam_tutar' => $toplam_tutar,
                'odenen_tutar' => $odenen_tutar,
                'kalan_tutar' => $toplam_tutar - $odenen_tutar,
                'message' => 'Satış işlemi başarıyla tamamlandı'
            ];
            
        } catch (Exception $e) {
            $this->database->rollback();
            error_log("Satış işlemi hatası: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Satış işlemi sırasında hata oluştu: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Stok alımı işlemi gerçekleştir
     * @param int $tedarikci_id Tedarikçi ID
     * @param array $urunler Ürün listesi [['urun_id' => 1, 'miktar' => 10, 'alis_fiyati' => 8.00], ...]
     * @param float $odenen_tutar Ödenen tutar
     * @param string $aciklama Açıklama
     * @return array İşlem sonucu
     */
    public function processStockPurchase($tedarikci_id, $urunler, $odenen_tutar = 0, $aciklama = '') {
        try {
            return $this->transactionManager->execute(function($db) use ($tedarikci_id, $urunler, $odenen_tutar, $aciklama) {
                // Toplam tutarı hesapla
                $toplam_tutar = 0;
                foreach ($urunler as $urun) {
                    $toplam_tutar += $urun['miktar'] * $urun['alis_fiyati'];
                }
                
                // Stok alımı için açıklama oluştur
                $aciklama_text = $aciklama ?: "Tedarikçi ID: $tedarikci_id - Stok alımı";
                $transfer_id = null; // Stok transfer tablosu kullanılmıyor
                
                // Stok hareketlerini ekle ve stokları güncelle
                foreach ($urunler as $urun) {
                    // Mevcut stok bilgisini al
                    $sql = "SELECT stok_miktari FROM urunler WHERE urun_id = ?";
                    $current = $this->database->fetch($sql, [$urun['urun_id']]);
                    $onceki_stok = $current['stok_miktari'];
                    $yeni_stok = $onceki_stok + $urun['miktar'];
                    
                    // Stok hareketi ekle
                    $sql = "INSERT INTO stok_hareketleri (urun_id, hareket_tipi, miktar, onceki_stok, yeni_stok, birim_fiyat, toplam_tutar, aciklama, kullanici_id) 
                            VALUES (?, 'giris', ?, ?, ?, ?, ?, ?, ?)";
                    $this->database->execute($sql, [
                        $urun['urun_id'], 
                        $urun['miktar'],
                        $onceki_stok,
                        $yeni_stok,
                        $urun['alis_fiyati'], 
                        $urun['miktar'] * $urun['alis_fiyati'],
                        $aciklama_text,
                        $_SESSION['user_id'] ?? 1
                    ]);
                    
                    // Stok güncelle
                    $sql = "UPDATE urunler SET stok_miktari = ?, alis_fiyati = ? WHERE urun_id = ?";
                    $this->database->execute($sql, [$yeni_stok, $urun['alis_fiyati'], $urun['urun_id']]);
                }
                
                // Cari hareketleri ekle (Tedarikçiye borçlanma)
                // BORÇ kaydı (biz tedarikçiye borçluyuz)
                $sql = "INSERT INTO cari_hareketler (cari_id, hareket_tipi, tutar, aciklama) VALUES (?, ?, ?, ?)";
                $this->database->execute($sql, [$tedarikci_id, 'borc', $toplam_tutar, 
                    "Stok alımı - Transfer ID: $transfer_id" . ($aciklama ? " - $aciklama" : "")]);
                
                // Ödeme yapıldıysa ALACAK kaydı ekle (borcumuz azalıyor)
                if ($odenen_tutar > 0) {
                    $sql = "INSERT INTO cari_hareketler (cari_id, hareket_tipi, tutar, aciklama) VALUES (?, ?, ?, ?)";
                    $this->database->execute($sql, [$tedarikci_id, 'alacak', $odenen_tutar, 
                        "Stok alımı ödemesi - Transfer ID: $transfer_id" . ($aciklama ? " - $aciklama" : "")]);
                }
                
                // Bakiye güncelle (transaction içinde - aynı değer)
                $calculated_balance = $this->calculateBalance($tedarikci_id);
                $sql = "UPDATE cariler SET cari_bakiye = ?, bakiye = ? WHERE cari_id = ?";
                $this->database->execute($sql, [$calculated_balance, $calculated_balance, $tedarikci_id]);
                
                return [
                    'success' => true,
                    'transfer_id' => $transfer_id,
                    'toplam_tutar' => $toplam_tutar,
                    'odenen_tutar' => $odenen_tutar,
                    'kalan_borc' => $toplam_tutar - $odenen_tutar,
                    'message' => 'Stok alımı işlemi başarıyla tamamlandı'
                ];
            });
            
        } catch (Exception $e) {
            error_log("Stok alımı işlemi hatası: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Stok alımı işlemi sırasında hata oluştu: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Manuel ödeme işlemi gerçekleştir
     * @param int $cari_id Cari ID
     * @param float $tutar Ödeme tutarı
     * @param string $odeme_tipi 'tahsilat' (müşteriden) veya 'odeme' (tedarikçiye)
     * @param string $aciklama Açıklama
     * @return array İşlem sonucu
     */
    public function processManualPayment($cari_id, $tutar, $odeme_tipi, $aciklama = '') {
        try {
            // Cari bilgilerini al
            $sql = "SELECT firma_adi, cari_tipi, bakiye FROM cariler WHERE cari_id = ?";
            $cari = $this->database->fetch($sql, [$cari_id]);
            
            if (!$cari) {
                throw new Exception("Cari bulunamadı");
            }
            
            $eski_bakiye = $cari['bakiye'];
            
            // Ödeme tipine göre hareket tipi belirle
            if ($odeme_tipi === 'tahsilat') {
                // Müşteriden tahsilat - müşterinin borcu azalıyor - alacak kaydı
                $hareket_tipi = 'alacak';
                $aciklama = $aciklama ?: 'Manuel tahsilat';
            } else {
                // Tedarikçiye ödeme - bizim borcumuz azalıyor - alacak kaydı
                $hareket_tipi = 'alacak';
                $aciklama = $aciklama ?: 'Manuel ödeme';
            }
            
            // Cari hareket ekle ve bakiye güncelle
            $this->addMovementAndUpdateBalance($cari_id, $hareket_tipi, $tutar, $aciklama);
            
            // Güncel bakiyeyi al
            $yeni_bakiye = $this->calculateBalance($cari_id);
            
            return [
                'success' => true,
                'cari_adi' => $cari['firma_adi'],
                'odeme_tipi' => $odeme_tipi,
                'tutar' => $tutar,
                'eski_bakiye' => $eski_bakiye,
                'yeni_bakiye' => $yeni_bakiye,
                'message' => ucfirst($odeme_tipi) . ' işlemi başarıyla tamamlandı'
            ];
            
        } catch (Exception $e) {
            error_log("Manuel ödeme işlemi hatası: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Ödeme işlemi sırasında hata oluştu: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Satış iptali işlemi
     * @param int $satis_id Satış ID
     * @param string $aciklama İptal açıklaması
     * @return array İşlem sonucu
     */
    public function cancelSale($satis_id, $aciklama = '') {
        try {
            $this->database->beginTransaction();
            
            // Satış bilgilerini al
            $sql = "SELECT * FROM satislar WHERE satis_id = ?";
            $satis = $this->database->fetch($sql, [$satis_id]);
            
            if (!$satis) {
                throw new Exception("Satış bulunamadı");
            }
            
            // Satış detaylarını al (stok iadesi için)
            $sql = "SELECT * FROM satis_detay WHERE satis_id = ?";
            $detaylar = $this->database->fetchAll($sql, [$satis_id]);
            
            // Stokları iade et
            foreach ($detaylar as $detay) {
                $sql = "UPDATE urunler SET stok_miktari = stok_miktari + ? WHERE urun_id = ?";
                $this->database->execute($sql, [$detay['miktar'], $detay['urun_id']]);
            }
            
            // Ters cari hareketleri ekle
            $aciklama = $aciklama ?: 'Satış iptali';
            
            // Borç kaydını iptal et (alacak kaydı ekle)
            $this->addMovementAndUpdateBalance($satis['cari_id'], 'alacak', $satis['toplam_tutar'], 
                "Satış iptali - Satış No: $satis_id - $aciklama");
            
            // Ödeme varsa onu da iptal et (borç kaydı ekle)
            if ($satis['odenen_tutar'] > 0) {
                $this->addMovementAndUpdateBalance($satis['cari_id'], 'borc', $satis['odenen_tutar'], 
                    "Satış iptali - Ödeme iadesi - Satış No: $satis_id - $aciklama");
            }
            
            // Satışı iptal olarak işaretle (silmek yerine)
            $sql = "UPDATE satislar SET aciklama = CONCAT(COALESCE(aciklama, ''), ' - İPTAL EDİLDİ: $aciklama') WHERE satis_id = ?";
            $this->database->execute($sql, [$satis_id]);
            
            $this->database->commit();
            
            return [
                'success' => true,
                'satis_id' => $satis_id,
                'iptal_tutari' => $satis['toplam_tutar'],
                'message' => 'Satış başarıyla iptal edildi'
            ];
            
        } catch (Exception $e) {
            $this->database->rollback();
            error_log("Satış iptali hatası: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Satış iptali sırasında hata oluştu: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Finansal işlem geçmişini getir
     * @param int $cari_id Cari ID (opsiyonel)
     * @param int $limit Kayıt limiti
     * @return array İşlem geçmişi
     */
    public function getTransactionHistory($cari_id = null, $limit = 50) {
        try {
            $sql = "SELECT 
                        ch.hareket_id,
                        ch.cari_id,
                        c.firma_adi,
                        ch.hareket_tipi,
                        ch.tutar,
                        ch.aciklama,
                        ch.satis_id,
                        ch.tarih
                    FROM cari_hareketler ch
                    JOIN cariler c ON ch.cari_id = c.cari_id";
            
            $params = [];
            if ($cari_id) {
                $sql .= " WHERE ch.cari_id = ?";
                $params[] = $cari_id;
            }
            
            $sql .= " ORDER BY ch.tarih DESC LIMIT ?";
            $params[] = $limit;
            
            return $this->database->fetchAll($sql, $params);
            
        } catch (Exception $e) {
            error_log("İşlem geçmişi hatası: " . $e->getMessage());
            return [];
        }
    }
}
?>

