<?php
/**
 * Veritabanı Analiz Web Interface
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../includes/data_cleanup.php';
require_once __DIR__ . '/../includes/balance_manager.php';
require_once __DIR__ . '/../includes/security.php';

requireLogin();

// Yetki kontrolü geçici olarak devre dışı (analiz için)
// if (!isset($_SESSION['rol_id']) || $_SESSION['rol_id'] != 1) {
//     die("Bu sayfaya erişim yetkiniz yok.");
// }

$page_title = "Veritabanı Analizi";

try {
    $database = Database::getInstance();
    $dataCleanup = new DataCleanup($database);
    $balanceManager = new BalanceManager($database);
    
    // 1. Veri Tipi Kontrolü
    $tables = ['cariler', 'cari_hareketler', 'satislar', 'stok_hareketleri', 'urunler'];
    $dataTypeIssues = [];
    $dataTypeResults = [];
    
    foreach ($tables as $table) {
        $sql = "SHOW COLUMNS FROM $table";
        $columns = $database->fetchAll($sql);
        
        foreach ($columns as $column) {
            if (strpos($column['Type'], 'decimal') !== false) {
                $isCorrect = strpos($column['Type'], 'decimal(15,2)') !== false;
                $dataTypeResults[] = [
                    'table' => $table,
                    'column' => $column['Field'],
                    'type' => $column['Type'],
                    'correct' => $isCorrect
                ];
                
                if (!$isCorrect) {
                    $dataTypeIssues[] = [
                        'table' => $table,
                        'column' => $column['Field'],
                        'current_type' => $column['Type']
                    ];
                }
            }
        }
    }
    
    // 2. Bakiye Tutarsızlıkları
    $balanceIssues = $balanceManager->checkBalanceConsistency();
    
    // 3. Negatif Müşteri Bakiyeleri
    $sql = "SELECT cari_id, firma_adi, cari_adi, cari_bakiye, bakiye 
            FROM cariler 
            WHERE cari_tipi = 'musteri' AND (cari_bakiye < 0 OR bakiye > 0)";
    $negativeCustomers = $database->fetchAll($sql);
    
    // 4. Overflow Kayıtlar
    $overflowRecords = $dataCleanup->findOverflowRecords();
    $totalOverflow = array_sum(array_map('count', $overflowRecords));
    
    // 5. Orphaned Kayıtlar
    $sql = "SELECT ch.hareket_id, ch.cari_id 
            FROM cari_hareketler ch 
            LEFT JOIN cariler c ON ch.cari_id = c.cari_id 
            WHERE c.cari_id IS NULL";
    $orphanedHareketler = $database->fetchAll($sql);
    
    $sql = "SELECT s.satis_id, s.cari_id 
            FROM satislar s 
            LEFT JOIN cariler c ON s.cari_id = c.cari_id 
            WHERE c.cari_id IS NULL";
    $orphanedSatislar = $database->fetchAll($sql);
    
    $totalOrphaned = count($orphanedHareketler) + count($orphanedSatislar);
    
    // 6. Satış Tutarsızlıkları
    $sql = "SELECT satis_id, toplam_tutar, odenen_tutar, kalan_tutar,
                   (toplam_tutar - odenen_tutar) as calculated_kalan
            FROM satislar 
            WHERE ABS(kalan_tutar - (toplam_tutar - odenen_tutar)) > 0.01";
    $salesIssues = $database->fetchAll($sql);
    
    // 7. Genel İstatistikler
    $stats = [
        'Toplam Cari' => $database->fetch("SELECT COUNT(*) as count FROM cariler")['count'],
        'Toplam Müşteri' => $database->fetch("SELECT COUNT(*) as count FROM cariler WHERE cari_tipi = 'musteri'")['count'],
        'Toplam Tedarikçi' => $database->fetch("SELECT COUNT(*) as count FROM cariler WHERE cari_tipi = 'tedarikci'")['count'],
        'Toplam Cari Hareket' => $database->fetch("SELECT COUNT(*) as count FROM cari_hareketler")['count'],
        'Toplam Satış' => $database->fetch("SELECT COUNT(*) as count FROM satislar")['count'],
        'Toplam Stok Hareketi' => $database->fetch("SELECT COUNT(*) as count FROM stok_hareketleri")['count'],
        'Toplam Ürün' => $database->fetch("SELECT COUNT(*) as count FROM urunler")['count']
    ];
    
    // 8. Bakiye Özeti
    $balanceSummary = $balanceManager->getBalanceSummary();
    
    // Toplam sorun sayısı
    $totalIssues = count($dataTypeIssues) + count($balanceIssues) + count($negativeCustomers) + 
                   $totalOverflow + $totalOrphaned + count($salesIssues);
    
} catch (Exception $e) {
    $error = $e->getMessage();
}

?>
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $page_title; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .issue-card { border-left: 4px solid #dc3545; }
        .success-card { border-left: 4px solid #28a745; }
        .warning-card { border-left: 4px solid #ffc107; }
        .stat-card { transition: transform 0.2s; }
        .stat-card:hover { transform: translateY(-5px); }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row mb-4">
            <div class="col-12">
                <h1><i class="fas fa-database me-2"></i><?php echo $page_title; ?></h1>
                <p class="text-muted">Veritabanı durumu ve tespit edilen sorunlar</p>
            </div>
        </div>

        <?php if (isset($error)): ?>
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Hata:</strong> <?php echo htmlspecialchars($error); ?>
            </div>
        <?php else: ?>

        <!-- Özet Kartlar -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card stat-card <?php echo $totalIssues > 0 ? 'issue-card' : 'success-card'; ?>">
                    <div class="card-body text-center">
                        <i class="fas fa-exclamation-circle fa-3x mb-3 <?php echo $totalIssues > 0 ? 'text-danger' : 'text-success'; ?>"></i>
                        <h2><?php echo $totalIssues; ?></h2>
                        <p class="mb-0">Toplam Sorun</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card stat-card <?php echo count($dataTypeIssues) > 0 ? 'warning-card' : 'success-card'; ?>">
                    <div class="card-body text-center">
                        <i class="fas fa-database fa-3x mb-3 <?php echo count($dataTypeIssues) > 0 ? 'text-warning' : 'text-success'; ?>"></i>
                        <h2><?php echo count($dataTypeIssues); ?></h2>
                        <p class="mb-0">Veri Tipi Sorunu</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card stat-card <?php echo count($balanceIssues) > 0 ? 'issue-card' : 'success-card'; ?>">
                    <div class="card-body text-center">
                        <i class="fas fa-balance-scale fa-3x mb-3 <?php echo count($balanceIssues) > 0 ? 'text-danger' : 'text-success'; ?>"></i>
                        <h2><?php echo count($balanceIssues); ?></h2>
                        <p class="mb-0">Bakiye Tutarsızlığı</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card stat-card <?php echo count($negativeCustomers) > 0 ? 'issue-card' : 'success-card'; ?>">
                    <div class="card-body text-center">
                        <i class="fas fa-user-times fa-3x mb-3 <?php echo count($negativeCustomers) > 0 ? 'text-danger' : 'text-success'; ?>"></i>
                        <h2><?php echo count($negativeCustomers); ?></h2>
                        <p class="mb-0">Negatif Müşteri</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- 1. Veri Tipi Kontrolü -->
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="fas fa-table me-2"></i>1. Veri Tipi Kontrolü</h5>
            </div>
            <div class="card-body">
                <?php if (empty($dataTypeIssues)): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle me-2"></i>
                        Tüm veri tipleri doğru (DECIMAL(15,2))
                    </div>
                <?php else: ?>
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo count($dataTypeIssues); ?> veri tipi sorunu bulundu
                    </div>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Tablo</th>
                                    <th>Kolon</th>
                                    <th>Mevcut Tip</th>
                                    <th>Durum</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($dataTypeResults as $result): ?>
                                <tr>
                                    <td><?php echo $result['table']; ?></td>
                                    <td><?php echo $result['column']; ?></td>
                                    <td><code><?php echo $result['type']; ?></code></td>
                                    <td>
                                        <?php if ($result['correct']): ?>
                                            <span class="badge bg-success">✓ Doğru</span>
                                        <?php else: ?>
                                            <span class="badge bg-danger">✗ Yanlış</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- 2. Bakiye Tutarsızlıkları -->
        <div class="card mb-4">
            <div class="card-header bg-danger text-white">
                <h5 class="mb-0"><i class="fas fa-balance-scale me-2"></i>2. Bakiye Tutarsızlıkları</h5>
            </div>
            <div class="card-body">
                <?php if (empty($balanceIssues)): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle me-2"></i>
                        Tüm bakiyeler tutarlı
                    </div>
                <?php else: ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo count($balanceIssues); ?> bakiye tutarsızlığı bulundu
                    </div>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Cari ID</th>
                                    <th>Firma Adı</th>
                                    <th>Kayıtlı Bakiye</th>
                                    <th>Hesaplanan Bakiye</th>
                                    <th>Fark</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($balanceIssues as $issue): ?>
                                <tr>
                                    <td><?php echo $issue['cari_id']; ?></td>
                                    <td><?php echo htmlspecialchars($issue['firma_adi']); ?></td>
                                    <td><?php echo number_format($issue['stored_balance'], 2); ?> TL</td>
                                    <td><?php echo number_format($issue['calculated_balance'], 2); ?> TL</td>
                                    <td class="text-danger">
                                        <strong><?php echo number_format(abs($issue['stored_balance'] - $issue['calculated_balance']), 2); ?> TL</strong>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- 3. Negatif Müşteri Bakiyeleri -->
        <div class="card mb-4">
            <div class="card-header bg-warning text-dark">
                <h5 class="mb-0"><i class="fas fa-user-times me-2"></i>3. Negatif Müşteri Bakiyeleri</h5>
            </div>
            <div class="card-body">
                <?php if (empty($negativeCustomers)): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle me-2"></i>
                        Negatif müşteri bakiyesi yok
                    </div>
                <?php else: ?>
                    <div class="alert alert-warning">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        <?php echo count($negativeCustomers); ?> negatif müşteri bakiyesi bulundu (HATA!)
                    </div>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>Cari ID</th>
                                    <th>Firma Adı</th>
                                    <th>Cari Bakiye</th>
                                    <th>Bakiye</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($negativeCustomers as $customer): ?>
                                <tr>
                                    <td><?php echo $customer['cari_id']; ?></td>
                                    <td><?php echo htmlspecialchars($customer['firma_adi']); ?></td>
                                    <td class="text-danger"><?php echo number_format($customer['cari_bakiye'], 2); ?> TL</td>
                                    <td class="text-danger"><?php echo number_format($customer['bakiye'], 2); ?> TL</td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- 4-6. Diğer Sorunlar -->
        <div class="row mb-4">
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header bg-info text-white">
                        <h6 class="mb-0">4. Overflow Kayıtlar</h6>
                    </div>
                    <div class="card-body">
                        <?php if ($totalOverflow === 0): ?>
                            <p class="text-success mb-0"><i class="fas fa-check-circle me-2"></i>Overflow kayıt yok</p>
                        <?php else: ?>
                            <p class="text-danger mb-2"><strong><?php echo $totalOverflow; ?> overflow kayıt bulundu</strong></p>
                            <ul class="mb-0">
                                <?php foreach ($overflowRecords as $table => $records): ?>
                                <li><?php echo $table; ?>: <?php echo count($records); ?> kayıt</li>
                                <?php endforeach; ?>
                            </ul>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header bg-secondary text-white">
                        <h6 class="mb-0">5. Orphaned Kayıtlar</h6>
                    </div>
                    <div class="card-body">
                        <?php if ($totalOrphaned === 0): ?>
                            <p class="text-success mb-0"><i class="fas fa-check-circle me-2"></i>Orphaned kayıt yok</p>
                        <?php else: ?>
                            <p class="text-danger mb-2"><strong><?php echo $totalOrphaned; ?> orphaned kayıt bulundu</strong></p>
                            <ul class="mb-0">
                                <li>cari_hareketler: <?php echo count($orphanedHareketler); ?> kayıt</li>
                                <li>satislar: <?php echo count($orphanedSatislar); ?> kayıt</li>
                            </ul>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header bg-dark text-white">
                        <h6 class="mb-0">6. Satış Tutarsızlıkları</h6>
                    </div>
                    <div class="card-body">
                        <?php if (empty($salesIssues)): ?>
                            <p class="text-success mb-0"><i class="fas fa-check-circle me-2"></i>Tüm satışlar tutarlı</p>
                        <?php else: ?>
                            <p class="text-danger mb-0"><strong><?php echo count($salesIssues); ?> satış tutarsızlığı bulundu</strong></p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- 7. Genel İstatistikler -->
        <div class="card mb-4">
            <div class="card-header bg-success text-white">
                <h5 class="mb-0"><i class="fas fa-chart-bar me-2"></i>7. Genel İstatistikler</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <?php foreach ($stats as $label => $value): ?>
                    <div class="col-md-3 mb-3">
                        <div class="border rounded p-3 text-center">
                            <h4 class="text-primary mb-0"><?php echo number_format($value); ?></h4>
                            <small class="text-muted"><?php echo $label; ?></small>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <!-- 8. Bakiye Özeti -->
        <div class="card mb-4">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="fas fa-wallet me-2"></i>8. Bakiye Özeti</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <ul class="list-group">
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Toplam Cari:</span>
                                <strong><?php echo number_format($balanceSummary['toplam_cari']); ?></strong>
                            </li>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Borçlu Cari:</span>
                                <strong class="text-danger"><?php echo number_format($balanceSummary['borclu_cari']); ?></strong>
                            </li>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Alacaklı Cari:</span>
                                <strong class="text-success"><?php echo number_format($balanceSummary['alacakli_cari']); ?></strong>
                            </li>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Sıfır Bakiye:</span>
                                <strong><?php echo number_format($balanceSummary['sifir_bakiye']); ?></strong>
                            </li>
                        </ul>
                    </div>
                    <div class="col-md-6">
                        <ul class="list-group">
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Toplam Borç:</span>
                                <strong class="text-danger"><?php echo number_format($balanceSummary['toplam_borc'], 2); ?> TL</strong>
                            </li>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Toplam Alacak:</span>
                                <strong class="text-success"><?php echo number_format($balanceSummary['toplam_alacak'], 2); ?> TL</strong>
                            </li>
                            <li class="list-group-item d-flex justify-content-between">
                                <span>Net Bakiye:</span>
                                <strong class="text-primary"><?php echo number_format($balanceSummary['net_bakiye'], 2); ?> TL</strong>
                            </li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>

        <!-- Özet Rapor -->
        <div class="card">
            <div class="card-header <?php echo $totalIssues > 0 ? 'bg-danger' : 'bg-success'; ?> text-white">
                <h5 class="mb-0"><i class="fas fa-clipboard-check me-2"></i>ÖZET RAPOR</h5>
            </div>
            <div class="card-body">
                <?php if ($totalIssues === 0): ?>
                    <div class="alert alert-success mb-0">
                        <h4><i class="fas fa-check-circle me-2"></i>Veritabanı Temiz!</h4>
                        <p class="mb-0">Hiçbir sorun tespit edilmedi. Sistem sağlıklı çalışıyor.</p>
                    </div>
                <?php else: ?>
                    <div class="alert alert-danger mb-3">
                        <h4><i class="fas fa-exclamation-triangle me-2"></i>Toplam <?php echo $totalIssues; ?> Sorun Tespit Edildi</h4>
                        <ul class="mb-0">
                            <li>Veri tipi sorunları: <strong><?php echo count($dataTypeIssues); ?></strong></li>
                            <li>Bakiye tutarsızlıkları: <strong><?php echo count($balanceIssues); ?></strong></li>
                            <li>Negatif müşteri bakiyeleri: <strong><?php echo count($negativeCustomers); ?></strong></li>
                            <li>Overflow kayıtlar: <strong><?php echo $totalOverflow; ?></strong></li>
                            <li>Orphaned kayıtlar: <strong><?php echo $totalOrphaned; ?></strong></li>
                            <li>Satış tutarsızlıkları: <strong><?php echo count($salesIssues); ?></strong></li>
                        </ul>
                    </div>
                    <div class="alert alert-warning mb-0">
                        <h5><i class="fas fa-tools me-2"></i>MİGRATION GEREKLİ!</h5>
                        <p class="mb-0">Veritabanı migration işlemi yapılmalıdır. Lütfen yönetici ile iletişime geçin.</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <?php endif; ?>

        <div class="mt-4 text-center">
            <a href="../pages/dashboard.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left me-2"></i>Dashboard'a Dön
            </a>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
